<?php

namespace App\Console\Commands;

use App\Facades\Vrm\MediaForge;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class MediaForgeCompressCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'mediaforge:compress
                            {directory? : Directory under public path (e.g. media/zidii)}
                            {--quality= : Compression quality 1-100 (default from config)}
                            {--override : Replace originals with compressed version}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Compress raster images in a directory using MediaForge';

    /**
     * Raster image extensions supported by MediaForge.
     *
     * @var array<int, string>
     */
    protected static array $rasterExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $directory = $this->argument('directory') ?? 'media/zidii';
        $directory = trim($directory, '/');
        $quality = $this->option('quality') !== null ? (int) $this->option('quality') : null;
        $override = $this->option('override');

        $basePath = public_path($directory);
        $publicRoot = realpath(public_path()) ?: public_path();

        if (! is_dir($basePath)) {
            $this->error("Directory does not exist: {$basePath}");

            return self::FAILURE;
        }

        $resolvedBase = realpath($basePath);
        if ($resolvedBase === false || ! str_starts_with($resolvedBase, $publicRoot)) {
            $this->error('Directory must be under the public path.');

            return self::FAILURE;
        }

        $paths = $this->collectImagePaths($resolvedBase);

        if (empty($paths)) {
            $this->info("No raster images found in {$directory}.");

            return self::SUCCESS;
        }

        $this->info('Compressing '.count($paths).' image(s)...');

        $result = MediaForge::compressExisting($paths, $quality, null, $override);

        foreach ($result['processed'] as $path) {
            $this->line("  <info>Processed:</info> {$path}");
        }
        foreach ($result['skipped'] as $msg) {
            $this->line("  <comment>Skipped:</comment> {$msg}");
        }
        foreach ($result['errors'] as $msg) {
            $this->error("  {$msg}");
        }

        $this->newLine();
        $this->info('Summary: '.count($result['processed']).' processed, '.count($result['skipped']).' skipped, '.count($result['errors']).' errors.');

        return count($result['errors']) > 0 ? self::FAILURE : self::SUCCESS;
    }

    /**
     * Collect relative paths (from public) to raster images under the given directory.
     *
     * @return array<int, string>
     */
    protected function collectImagePaths(string $fullPath): array
    {
        $paths = [];
        $publicPath = public_path();

        foreach (File::allFiles($fullPath) as $file) {
            $ext = strtolower($file->getExtension());
            if (! in_array($ext, self::$rasterExtensions, true)) {
                continue;
            }
            $full = $file->getPathname();
            if (str_starts_with($full, $publicPath)) {
                $paths[] = substr($full, strlen($publicPath) + 1);
            }
        }

        return $paths;
    }
}
