<?php

namespace App\Jobs\V1;

use App\Jobs\Vrm\SendMail;
use App\Mail\VerifyEmailMail;
use App\Models\User;
use App\Services\Sms\SmsNotifier;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Log;
use Throwable;

class SendVerifyEmailJob implements ShouldBeUnique
{
    use Queueable;

    /**
     * Prevent duplicate verification sends for the same user within 2 minutes.
     */
    public int $uniqueFor = 120;

    protected User $user;

    protected string $verificationCode;

    /**
     * Create a new job instance.
     */
    public function __construct(User $user, string $verificationCode)
    {
        $this->user = $user;
        $this->verificationCode = $verificationCode;
    }

    /**
     * Unique job ID per user to prevent duplicate OTP SMS.
     */
    public function uniqueId(): string
    {
        return 'verify-email-'.$this->user->getKey();
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        app(SmsNotifier::class)->sendToPhone(
            $this->user->phone,
            "Your Ziidi Trader verification code is {$this->verificationCode}.",
            'user_otp'
        );

        try {
            SendMail::dispatch($this->user->email, mailable: new VerifyEmailMail($this->user->name, $this->verificationCode), notificationType: 'user_otp');
        } catch (Throwable $exception) {
            Log::warning('Verification email dispatch failed.', [
                'user_id' => $this->user->id,
                'email' => $this->user->email,
                'message' => $exception->getMessage(),
            ]);
        }

        Log::info('Verification code dispatched.', [
            'user_id' => $this->user->id,
            'email' => $this->user->email,
            'phone' => $this->user->phone,
            'verification_code' => $this->verificationCode,
        ]);
    }
}
