<?php

namespace App\Mail;

use App\Services\ReceiptPdfService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Attachment;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Queue\SerializesModels;

class WalletTransactionMail extends Mailable implements ShouldQueue
{
    use Queueable;
    use SerializesModels;

    public function __construct(
        public string $recipientName,
        public string $type,
        public string $currencyCode,
        public float $amount,
        public string $status = 'completed',
        public ?string $phoneNumber = null,
        public ?string $reference = null,
        public ?string $occurredAt = null,
        public ?string $message = null,
    ) {}

    public function envelope(): Envelope
    {
        $typeLabel = ucfirst(str_replace('_', ' ', $this->type));
        $statusLabel = ucfirst($this->status);

        return new Envelope(
            subject: "Wallet {$typeLabel}: {$statusLabel}",
        );
    }

    public function content(): Content
    {
        $typeLabel = ucfirst(str_replace('_', ' ', $this->type));
        $logoUrl = rtrim((string) config('app.url'), '/').'/media/zidii/assets/img/ziidi3.png';

        return new Content(
            view: 'emails.receipts.wallet-transaction',
            with: [
                'recipientName' => $this->recipientName,
                'typeLabel' => $typeLabel,
                'currencyCode' => $this->currencyCode,
                'amount' => $this->amount,
                'status' => $this->status,
                'phoneNumber' => $this->phoneNumber,
                'reference' => $this->reference,
                'occurredAt' => $this->occurredAt,
                'actionUrl' => app()->bound('url')
                    ? route('member.account')
                    : rtrim((string) config('app.url'), '/').'/member/account',
                'customMessage' => $this->message,
                'logoUrl' => $logoUrl,
            ],
        );
    }

    public function attachments(): array
    {
        if (! $this->shouldAttachReceipt()) {
            return [];
        }

        $type = str_replace(' ', '-', strtolower(str_replace('_', ' ', $this->type)));
        $timestamp = now()->format('YmdHis');

        return [
            Attachment::fromData(fn (): string => $this->receiptContent(), "receipt-{$type}-{$timestamp}.pdf")
                ->withMime('application/pdf'),
        ];
    }

    protected function shouldAttachReceipt(): bool
    {
        $type = strtolower($this->type);
        $status = strtolower($this->status);
        $isSuccessful = in_array($status, ['approved', 'completed', 'successful', 'success'], true);

        return (str_contains($type, 'deposit') || str_contains($type, 'withdraw')) && $isSuccessful;
    }

    protected function receiptContent(): string
    {
        return app(ReceiptPdfService::class)->generateWalletTransactionReceiptPdf(
            fullName: $this->recipientName,
            type: $this->type,
            currencyCode: $this->currencyCode,
            amount: $this->amount,
            reference: $this->reference,
            phoneOrBankAccountNumber: $this->phoneNumber,
            occurredAt: $this->occurredAt,
        );
    }
}
