<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminTransactionNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        protected string $memberName,
        protected string $memberEmail,
        protected float $amount,
        protected string $currencyCode,
        protected string $transactionType,
        protected string $status,
        protected string $channel = 'internal',
        protected ?string $reference = null,
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $typeLabel = ucfirst(str_replace('_', ' ', $this->transactionType));
        $statusLabel = ucfirst($this->status);
        $channelLabel = strtoupper($this->channel);

        return (new MailMessage)
            ->subject("{$typeLabel} {$statusLabel}: {$this->currencyCode} ".number_format($this->amount, 2))
            ->greeting('Hello Admin,')
            ->line("A member {$typeLabel} transaction has been {$this->status}.")
            ->line("Member: {$this->memberName} ({$this->memberEmail})")
            ->line("Channel: {$channelLabel}")
            ->line("Amount: {$this->currencyCode} ".number_format($this->amount, 2))
            ->line("Status: {$statusLabel}")
            ->line('Reference: '.($this->reference ?: 'N/A'))
            ->action('Open Transactions', route('admin.transactions.index'))
            ->line('Please review this transaction in the admin panel.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'member_name' => $this->memberName,
            'member_email' => $this->memberEmail,
            'amount' => $this->amount,
            'currency_code' => $this->currencyCode,
            'transaction_type' => $this->transactionType,
            'status' => $this->status,
            'channel' => $this->channel,
            'reference' => $this->reference,
        ];
    }
}
