<?php

namespace App\Providers;

use App\Listeners\ActivateUserOnEmailVerified;
use App\Services\DatabaseNSE;
use App\Services\NSEService;
use App\Services\RapidNSE;
use App\Services\YahooNSE;
use Carbon\CarbonImmutable;
use Illuminate\Auth\Events\Verified;
use Illuminate\Support\Facades\Date;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\ServiceProvider;
use Illuminate\Validation\Rules\Password;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        $this->app->bind(NSEService::class, function () {
            $provider = strtolower((string) config('services.nse_provider', 'rapid'));

            return match ($provider) {
                'database', 'db', 'local' => app(DatabaseNSE::class),
                'yahoo' => app(YahooNSE::class),
                default => app(RapidNSE::class),
            };
        });

        $this->app->bind(\App\Services\Sms\SmsProvider::class, function () {
            $provider = strtolower((string) config('services.sms_provider', 'africastalking'));

            return $provider === 'infobip'
                ? app(\App\Services\Sms\InfobipSms::class)
                : app(\App\Services\Sms\SmsAfrikasTalking::class);
        });
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        $this->configureDefaults();
        $this->registerEventListeners();
    }

    /**
     * Register event listeners for auth and other application events.
     */
    protected function registerEventListeners(): void
    {
        Event::listen(Verified::class, ActivateUserOnEmailVerified::class);
    }

    /**
     * Configure default behaviors for production-ready applications.
     */
    protected function configureDefaults(): void
    {
        Date::use(CarbonImmutable::class);

        DB::prohibitDestructiveCommands(
            app()->isProduction(),
        );

        Password::defaults(fn (): ?Password => app()->isProduction()
            ? Password::min(12)
                ->mixedCase()
                ->letters()
                ->numbers()
                ->symbols()
                ->uncompromised()
            : null
        );
    }
}
