<?php

namespace App\Services\Investment;

use App\Models\Investment;
use App\Models\InvestmentPackage;
use App\Models\User;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Services\DefaultCurrencyService;
use Illuminate\Support\Facades\DB;

class SubscriptionService
{
    public const string META_INTENT_SUBSCRIPTION = 'subscription';

    /**
     * Create a subscription: debit wallet, create investment and investment_debit transaction.
     *
     * @throws \InvalidArgumentException when amount is outside package min/max or insufficient balance
     */
    public function createSubscription(User $user, Wallet $wallet, InvestmentPackage $package, float $amount): Investment
    {
        $min = (float) $package->min_amount;
        $max = (float) $package->max_amount;
        if ($amount < $min || $amount > $max) {
            throw new \InvalidArgumentException(__('Amount must be between :min and :max.', ['min' => $min, 'max' => $max]));
        }

        $available = (float) $wallet->balance - (float) $wallet->used_amount;
        if ($available < $amount) {
            throw new \InvalidArgumentException(__('Insufficient available balance.'));
        }

        $expectedReturn = $package->calculateExpectedReturn($amount);
        $durationHours = $package->duration_hours;
        $startedAt = now();
        $maturesAt = (clone $startedAt)->addHours($durationHours);

        return DB::transaction(function () use ($user, $wallet, $package, $amount, $expectedReturn, $durationHours, $startedAt, $maturesAt): Investment {
            $wallet->balance = max(0, (float) $wallet->balance - $amount);
            $wallet->save();

            $investment = Investment::query()->create([
                'user_id' => $user->id,
                'wallet_id' => $wallet->id,
                'package_id' => $package->id,
                'principal_amount' => $amount,
                'expected_return_amount' => $expectedReturn,
                'currency_code' => app(DefaultCurrencyService::class)->code(),
                'duration_days' => (int) ceil($durationHours / 24),
                'duration_hours' => $durationHours,
                'started_at' => $startedAt,
                'matures_at' => $maturesAt,
                'status' => 'active',
                'is_active' => true,
            ]);

            WalletTransaction::query()->create([
                'wallet_id' => $wallet->id,
                'user_id' => $user->id,
                'type' => 'investment_debit',
                'status' => 'completed',
                'amount' => $amount,
                'currency_code' => app(DefaultCurrencyService::class)->code(),
                'description' => __('Subscription to :name', ['name' => $package->name]),
                'meta' => [
                    'investment_id' => $investment->id,
                    'package_id' => $package->id,
                    'duration_hours' => $durationHours,
                    'expected_return' => $expectedReturn,
                ],
            ]);

            return $investment;
        });
    }
}
