<?php

namespace App\Services;

use App\Models\Security;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Throwable;

class LiveStockService
{
    public function __construct(
        private readonly NSEService $nseService
    ) {}

    /**
     * Return all stocks from NSE API as-is (for API consumers).
     *
     * @return array<int, mixed>
     */
    public function getAllStocks(): array
    {
        return $this->nseService->getStocks();
    }

    /**
     * Fetch stock data by symbol from the Nairobi Stock Exchange RapidAPI.
     *
     * @param  string  $symbol  Stock symbol as expected by the API (e.g. "Safaricom")
     * @return array<string, mixed> Decoded JSON response
     *
     * @throws RequestException When the API returns a non-2xx response
     */
    public function getStock(string $symbol): array
    {
        return $this->nseService->getStock($symbol);
    }

    /**
     * Resolve current share price for a ticker using /stocks/{symbol}.
     * Falls back to provided value when API/config is unavailable.
     */
    public function getCurrentPriceForTicker(string $ticker, float $fallback = 0.0): float
    {
        $symbol = strtoupper(trim($ticker));
        if ($symbol === '') {
            return $fallback;
        }

        try {
            $raw = $this->getStock($symbol);
            $rows = $this->extractStockRows($raw);
            $first = is_array($rows[0] ?? null) ? $rows[0] : [];
            $price = $this->toFloat($first['price'] ?? $first['Price'] ?? 0);

            return $price > 0 ? $price : $fallback;
        } catch (Throwable $e) {
            Log::warning('Failed to resolve live share price', [
                'ticker' => $symbol,
                'message' => $e->getMessage(),
            ]);

            return $fallback;
        }
    }

    /**
     * Return securities for display (marquee etc.): live price/change from API,
     * id and logo from Security when ticker matches trading_name.
     *
     * @return array<int, array{id: int|null, symbol: string, name: string, price: float, change: float, logo: string|null, initials: string}>
     */
    public function getSecuritiesForDisplay(?int $limit = null): array
    {
        try {
            $raw = $this->nseService->getStocks();
            $rows = $this->extractStockRows($raw);
        } catch (Throwable $e) {
            Log::warning('Failed to fetch live stocks.', [
                'message' => $e->getMessage(),
            ]);

            return [];
        }

        if ($rows === []) {
            Log::error('Live stock data is empty', ['raw_keys' => array_keys($raw)]);

            return [];
        }

        $items = collect($rows)
            ->when($limit !== null, fn ($collection) => $collection->take($limit))
            ->map(function (mixed $item): array {
                $arr = is_array($item) ? $item : [];
                $symbol = strtoupper(trim((string) ($arr['ticker'] ?? $arr['symbol'] ?? $arr['Symbol'] ?? '')));
                $name = (string) ($arr['name'] ?? $arr['Name'] ?? $symbol);
                $price = $this->toFloat($arr['price'] ?? $arr['Price'] ?? 0);
                $change = $this->extractChangeValue($arr['change'] ?? $arr['change_percent'] ?? $arr['Change'] ?? 0);

                return [
                    'symbol' => $symbol,
                    'name' => $name,
                    'price' => $price,
                    'change' => $change,
                ];
            })
            ->filter(fn (array $row): bool => $row['symbol'] !== '')
            ->values();

        if ($items->isEmpty()) {
            return [];
        }

        if ($this->shouldSyncFromProvider()) {
            $this->syncSecuritiesFromLiveRows($items->all());
        }

        $securitiesBySymbol = Security::query()
            ->where('is_active', true)
            ->get()
            ->keyBy(fn (Security $s) => strtoupper(trim((string) $s->trading_name)));

        return $items->map(function (array $row) use ($securitiesBySymbol): array {
            $security = $securitiesBySymbol->get($row['symbol']);

            return [
                'id' => $security?->id,
                'symbol' => $row['symbol'],
                'name' => $row['name'],
                'price' => $row['price'],
                'change' => $row['change'],
                'logo' => $security?->logo,
                'initials' => $this->buildInitials($row['symbol'], $row['name']),
            ];
        })->all();
    }

    /**
     * @param  array<int, array{symbol: string, name: string, price: float, change: float}>  $rows
     */
    private function syncSecuritiesFromLiveRows(array $rows): void
    {
        foreach ($rows as $row) {
            $symbol = strtoupper(trim((string) ($row['symbol'] ?? '')));
            if ($symbol === '') {
                continue;
            }

            $price = max(0.0001, (float) ($row['price'] ?? 0.0001));
            $name = trim((string) ($row['name'] ?? $symbol));

            $security = Security::query()->where('trading_name', $symbol)->first() ?? new Security;
            $security->trading_name = $symbol;
            $security->name = $name !== '' ? $name : $symbol;
            $security->currency_code = $security->currency_code ?: 'KSH';
            $security->initial_listing_amount = (float) ($security->initial_listing_amount ?: $price);
            $security->current_amount = $price;
            $security->is_active = true;
            $security->description = $security->description ?: Str::limit($security->name, 255);
            $security->save();
        }
    }

    /**
     * @param  array<int|string, mixed>  $raw
     * @return array<int, mixed>
     */
    private function extractStockRows(array $raw): array
    {
        $rows = $raw['data']['data'] ?? $raw['data'] ?? $raw;

        return is_array($rows) ? array_values($rows) : [];
    }

    private function toFloat(mixed $value): float
    {
        if (is_numeric($value)) {
            return (float) $value;
        }

        if (! is_string($value)) {
            return 0.0;
        }

        $normalized = str_replace(',', '', trim($value));

        return is_numeric($normalized) ? (float) $normalized : 0.0;
    }

    private function extractChangeValue(mixed $value): float
    {
        if (is_numeric($value)) {
            return (float) $value;
        }

        if (! is_string($value)) {
            return 0.0;
        }

        if (preg_match('/\(([+-]?\d+(?:\.\d+)?)%\)/', $value, $matches) === 1) {
            return (float) $matches[1];
        }

        if (preg_match('/([+-]?\d+(?:\.\d+)?)/', str_replace(',', '', $value), $matches) === 1) {
            return (float) $matches[1];
        }

        return 0.0;
    }

    private function buildInitials(string $symbol, string $name): string
    {
        $lettersOnly = preg_replace('/[^A-Z]/', '', strtoupper($symbol)) ?? '';

        if ($lettersOnly !== '') {
            return substr($lettersOnly, 0, 2);
        }

        $nameParts = preg_split('/\s+/', trim($name)) ?: [];
        $initials = collect($nameParts)
            ->filter()
            ->take(2)
            ->map(fn (string $part): string => strtoupper(substr($part, 0, 1)))
            ->implode('');

        return $initials !== '' ? $initials : 'ST';
    }

    private function shouldSyncFromProvider(): bool
    {
        $provider = strtolower((string) config('services.nse_provider', 'rapid'));

        return in_array($provider, ['rapid', 'yahoo'], true);
    }
}
