<?php

namespace App\Services;

use App\Models\Investment;
use App\Models\Notification;
use App\Models\NotificationState;
use App\Models\User;
use App\Models\WalletTransaction;

class MemberNotificationService
{
    public function notifyDepositRequested(WalletTransaction $transaction): void
    {
        $user = $transaction->user;

        if (! $user) {
            return;
        }

        $this->createNotification(
            user: $user,
            type: 'wallet.deposit.requested',
            title: __('Deposit Request'),
            body: __('Your deposit request of :currency :amount has been submitted.', [
                'currency' => $transaction->currency_code,
                'amount' => $this->formatAmount((float) $transaction->amount),
            ]),
            actionUrl: route('member.transactions'),
            data: [
                'transaction_id' => $transaction->id,
                'transaction_type' => $transaction->type,
            ],
        );
    }

    public function notifyDepositReviewed(WalletTransaction $transaction, string $status): void
    {
        $user = $transaction->user;

        if (! $user) {
            return;
        }

        $isApproved = strtolower($status) === 'approved';

        $this->createNotification(
            user: $user,
            type: $isApproved ? 'wallet.deposit.approved' : 'wallet.deposit.rejected',
            title: $isApproved ? __('Deposit Approved') : __('Deposit Rejected'),
            body: $isApproved
                ? __('Your deposit of :currency :amount has been approved.', [
                    'currency' => $transaction->currency_code,
                    'amount' => $this->formatAmount((float) $transaction->amount),
                ])
                : __('Your deposit of :currency :amount was rejected.', [
                    'currency' => $transaction->currency_code,
                    'amount' => $this->formatAmount((float) $transaction->amount),
                ]),
            actionUrl: route('member.transactions'),
            data: [
                'transaction_id' => $transaction->id,
                'transaction_type' => $transaction->type,
                'status' => $status,
            ],
            priority: $isApproved ? 1 : 2,
        );
    }

    public function notifyWithdrawalRequested(WalletTransaction $transaction): void
    {
        $user = $transaction->user;

        if (! $user) {
            return;
        }

        $this->createNotification(
            user: $user,
            type: 'wallet.withdrawal.requested',
            title: __('Withdrawal Request'),
            body: __('Your withdrawal request of :currency :amount has been submitted.', [
                'currency' => $transaction->currency_code,
                'amount' => $this->formatAmount((float) $transaction->amount),
            ]),
            actionUrl: route('member.transactions'),
            data: [
                'transaction_id' => $transaction->id,
                'transaction_type' => $transaction->type,
            ],
        );
    }

    public function notifyWithdrawalReviewed(WalletTransaction $transaction, string $status): void
    {
        $user = $transaction->user;

        if (! $user) {
            return;
        }

        $isApproved = strtolower($status) === 'approved';

        $this->createNotification(
            user: $user,
            type: $isApproved ? 'wallet.withdrawal.approved' : 'wallet.withdrawal.rejected',
            title: $isApproved ? __('Withdrawal Approved') : __('Withdrawal Rejected'),
            body: $isApproved
                ? __('Your withdrawal of :currency :amount has been approved.', [
                    'currency' => $transaction->currency_code,
                    'amount' => $this->formatAmount((float) $transaction->amount),
                ])
                : __('Your withdrawal of :currency :amount was rejected.', [
                    'currency' => $transaction->currency_code,
                    'amount' => $this->formatAmount((float) $transaction->amount),
                ]),
            actionUrl: route('member.transactions'),
            data: [
                'transaction_id' => $transaction->id,
                'transaction_type' => $transaction->type,
                'status' => $status,
            ],
            priority: $isApproved ? 1 : 2,
        );
    }

    public function notifyInvestmentCreated(Investment $investment): void
    {
        $user = $investment->user;

        if (! $user) {
            return;
        }

        $this->createNotification(
            user: $user,
            type: 'investment.created',
            title: __('Investment Created'),
            body: __('Your investment of :currency :amount has been created.', [
                'currency' => $investment->currency_code,
                'amount' => $this->formatAmount((float) $investment->principal_amount),
            ]),
            actionUrl: route('member.investments'),
            data: [
                'investment_id' => $investment->id,
                'package_id' => $investment->package_id,
            ],
        );
    }

    public function notifyInvestmentReviewed(Investment $investment, string $status): void
    {
        $user = $investment->user;

        if (! $user) {
            return;
        }

        $isApproved = strtolower($status) === 'approved';

        $this->createNotification(
            user: $user,
            type: $isApproved ? 'investment.approved' : 'investment.rejected',
            title: $isApproved ? __('Investment Approved') : __('Investment Rejected'),
            body: $isApproved
                ? __('Your investment of :currency :amount has been approved.', [
                    'currency' => $investment->currency_code,
                    'amount' => $this->formatAmount((float) $investment->principal_amount),
                ])
                : __('Your investment of :currency :amount was rejected.', [
                    'currency' => $investment->currency_code,
                    'amount' => $this->formatAmount((float) $investment->principal_amount),
                ]),
            actionUrl: route('member.investments'),
            data: [
                'investment_id' => $investment->id,
                'package_id' => $investment->package_id,
                'status' => $status,
            ],
            priority: $isApproved ? 1 : 2,
        );
    }

    public function notifyInvestmentReviewedForUser(
        User $user,
        float $amount,
        string $currencyCode,
        string $status,
        ?int $packageId = null,
    ): void {
        $isApproved = strtolower($status) === 'approved';

        $this->createNotification(
            user: $user,
            type: $isApproved ? 'investment.approved' : 'investment.rejected',
            title: $isApproved ? __('Investment Approved') : __('Investment Rejected'),
            body: $isApproved
                ? __('Your investment of :currency :amount has been approved.', [
                    'currency' => $currencyCode,
                    'amount' => $this->formatAmount($amount),
                ])
                : __('Your investment of :currency :amount was rejected.', [
                    'currency' => $currencyCode,
                    'amount' => $this->formatAmount($amount),
                ]),
            actionUrl: route('member.investments'),
            data: [
                'package_id' => $packageId,
                'amount' => $amount,
                'status' => $status,
            ],
            priority: $isApproved ? 1 : 2,
        );
    }

    public function notifySubscriptionRequested(User $user, float $amount, string $currencyCode, ?int $packageId = null): void
    {
        $this->createNotification(
            user: $user,
            type: 'subscription.requested',
            title: __('Subscription Request'),
            body: __('Your subscription request of :currency :amount has been submitted.', [
                'currency' => $currencyCode,
                'amount' => $this->formatAmount($amount),
            ]),
            actionUrl: route('member.investments'),
            data: [
                'package_id' => $packageId,
                'amount' => $amount,
            ],
        );
    }

    public function notifySubscriptionReviewed(User $user, float $amount, string $currencyCode, string $status, ?int $packageId = null): void
    {
        $isApproved = strtolower($status) === 'approved';

        $this->createNotification(
            user: $user,
            type: $isApproved ? 'subscription.approved' : 'subscription.rejected',
            title: $isApproved ? __('Subscription Approved') : __('Subscription Rejected'),
            body: $isApproved
                ? __('Your subscription of :currency :amount has been approved.', [
                    'currency' => $currencyCode,
                    'amount' => $this->formatAmount($amount),
                ])
                : __('Your subscription of :currency :amount was rejected.', [
                    'currency' => $currencyCode,
                    'amount' => $this->formatAmount($amount),
                ]),
            actionUrl: route('member.investments'),
            data: [
                'package_id' => $packageId,
                'amount' => $amount,
                'status' => $status,
            ],
            priority: $isApproved ? 1 : 2,
        );
    }

    public function notifyInvestmentCompleted(Investment $investment): void
    {
        $user = $investment->user;

        if (! $user) {
            return;
        }

        $this->createNotification(
            user: $user,
            type: 'investment.completed',
            title: __('Investment Completed'),
            body: __('Your investment has been marked as complete. Return credited: :currency :amount.', [
                'currency' => $investment->currency_code,
                'amount' => $this->formatAmount((float) $investment->expected_return_amount),
            ]),
            actionUrl: route('member.investments'),
            data: [
                'investment_id' => $investment->id,
                'expected_return' => (float) $investment->expected_return_amount,
            ],
            priority: 1,
        );
    }

    /**
     * @param  array<string, mixed>  $data
     */
    private function createNotification(
        User $user,
        string $type,
        string $title,
        string $body,
        string $actionUrl,
        array $data = [],
        ?int $priority = null,
    ): void {
        $notification = Notification::query()->create([
            'type' => $type,
            'title' => $title,
            'body' => $body,
            'data' => $data,
            'action_url' => $actionUrl,
            'priority' => $priority,
            'is_active' => true,
        ]);

        NotificationState::query()->create([
            'notification_id' => $notification->id,
            'user_id' => $user->id,
            'is_active' => true,
        ]);
    }

    private function formatAmount(float $amount): string
    {
        return number_format($amount, 2);
    }
}
