<?php

namespace App\Services\Payments;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

class MpesaService
{
    protected string $consumerKey;
    protected string $consumerSecret;
    protected string $shortCode;
    protected string $passkey;
    protected string $callbackUrl;
    protected string $baseUrl;

    public function __construct()
    {
        $this->consumerKey = config('services.mpesa.consumer_key');
        $this->consumerSecret = config('services.mpesa.consumer_secret');
        $this->shortCode = config('services.mpesa.short_code');
        $this->passkey = config('services.mpesa.passkey');
        $this->callbackUrl = config('services.mpesa.callback_url') ?: route('api.mpesa.callback');
        $this->baseUrl = config('services.mpesa.env') === 'production' 
            ? 'https://api.safaricom.co.ke' 
            : 'https://sandbox.safaricom.co.ke';
    }

    /**
     * Generate Access Token
     */
    public function getAccessToken(): ?string
    {
        try {
            $response = Http::withBasicAuth($this->consumerKey, $this->consumerSecret)
                ->get($this->baseUrl . '/oauth/v1/generate?grant_type=client_credentials');

            if ($response->successful()) {
                return $response->json('access_token');
            }

            Log::error('Mpesa Access Token Error', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
        } catch (Exception $e) {
            Log::error('Mpesa Access Token Exception: ' . $e->getMessage());
        }

        return null;
    }

    /**
     * Initiate STK Push
     */
    public function stkPush(string $phoneNumber, float $amount, string $reference, string $description = 'Payment'): array
    {
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return ['status' => false, 'message' => 'Could not generate access token'];
        }

        $timestamp = now()->format('YmdHis');
        $password = base64_encode($this->shortCode . $this->passkey . $timestamp);
        
        // Normalize phone number to 254XXXXXXXXX
        $phoneNumber = $this->normalizePhoneNumber($phoneNumber);

        $partyB = \App\Models\Vrm\Utility::where('key', 'wallet_till')->value('value') ?: $this->shortCode;

        $payload = [
            'BusinessShortCode' => $this->shortCode,
            'Password' => $password,
            'Timestamp' => $timestamp,
            'TransactionType' => 'CustomerBuyGoodsOnline',
            'Amount' => round($amount),
            'PartyA' => $phoneNumber,
            'PartyB' => $partyB,
            'PhoneNumber' => $phoneNumber,
            'CallBackURL' => $this->callbackUrl,
            'AccountReference' => $reference,
            'TransactionDesc' => $description,
        ];

        try {
            $response = Http::withToken($accessToken)
                ->post($this->baseUrl . '/mpesa/stkpush/v1/processrequest', $payload);

            if ($response->successful()) {
                return [
                    'status' => true,
                    'data' => $response->json(),
                ];
            }

            Log::error('Mpesa STK Push Error', [
                'status' => $response->status(),
                'body' => $response->body(),
                'payload' => $payload
            ]);

            return [
                'status' => false,
                'message' => $response->json('CustomerMessage') ?: $response->json('errorMessage') ?: 'STK Push failed',
            ];
        } catch (Exception $e) {
            Log::error('Mpesa STK Push Exception: ' . $e->getMessage());
            return ['status' => false, 'message' => 'STK Push exception occurred'];
        }
    }

    /**
     * Normalize phone number to 254XXXXXXXXX
     */
    public function normalizePhoneNumber(string $phone): string
    {
        $phone = preg_replace('/\D+/', '', $phone);

        if (str_starts_with($phone, '0')) {
            $phone = '254' . substr($phone, 1);
        } elseif (str_starts_with($phone, '7') || str_starts_with($phone, '1')) {
            $phone = '254' . $phone;
        } elseif (str_starts_with($phone, '+')) {
            $phone = substr($phone, 1);
        }

        return $phone;
    }
}
