<?php

namespace App\Services;

use App\Models\User;
use App\Models\WalletTransaction;
use Illuminate\Support\Carbon;
use Mpdf\Mpdf;
use Mpdf\Output\Destination;
use Throwable;

class ReceiptPdfService
{
    /**
     * Build receipt data array for a wallet transaction (for Blade receipt and email views).
     *
     * @return array{customerName: string, date: string, amount: string, phoneNumber: string, transactionNo: string, paymentType: string, maskedPhone: string}
     */
    public function receiptDataForTransaction(WalletTransaction $transaction, ?User $user = null): array
    {
        $user = $user ?? $transaction->user;
        $phone = $transaction->phone_number ?: $user?->phone;
        $occurredAt = $transaction->processed_at?->toDateTimeString()
            ?? $transaction->approved_at?->toDateTimeString()
            ?? $transaction->updated_at?->toDateTimeString();

        return [
            'customerName' => $user?->name ?? 'Customer',
            'date' => $this->formatDateTime($occurredAt),
            'amount' => number_format((float) $transaction->amount, 2),
            'phoneNumber' => $phone ?: 'N/A',
            'transactionNo' => $transaction->transaction_code ?: (string) $transaction->id,
            'paymentType' => $this->paymentTypeLabel((string) $transaction->type),
            'maskedPhone' => $this->maskPhoneNumber($phone),
        ];
    }

    protected function paymentTypeLabel(string $type): string
    {
        $normalized = strtolower($type);
        if (str_contains($normalized, 'withdraw')) {
            return 'Withdrawal';
        }
        if (str_contains($normalized, 'transfer')) {
            return 'Transfer';
        }
        if (str_contains($normalized, 'payment') || str_contains($normalized, 'order_debit') || str_contains($normalized, 'order_credit')) {
            return 'Payment';
        }

        return 'Deposit';
    }

    public function generateWalletTransactionReceiptPdf(
        string $fullName,
        string $type,
        string $currencyCode,
        float $amount,
        ?string $reference = null,
        ?string $phoneOrBankAccountNumber = null,
        ?string $occurredAt = null,
    ): string {
        $receipt = [
            'customerName' => $fullName,
            'date' => $this->formatDateTime($occurredAt),
            'amount' => number_format($amount, 2),
            'phoneNumber' => $phoneOrBankAccountNumber ?: 'N/A',
            'transactionNo' => $reference ?: 'N/A',
            'paymentType' => $this->paymentTypeLabel($type),
            'maskedPhone' => $this->maskPhoneNumber($phoneOrBankAccountNumber),
        ];

        $logoPath = public_path('media/zidii/assets/img/ziidi3.png');
        if (! file_exists($logoPath)) {
            $logoPath = null;
        }

        $html = view('pdf.receipt-preview', [
            'receipt' => $receipt,
            'logoPath' => $logoPath,
        ])->render();

        $mpdf = new Mpdf([
            'format' => 'A4',
            'margin_top' => 15,
            'margin_bottom' => 15,
            'margin_left' => 15,
            'margin_right' => 15,
        ]);

        $mpdf->SetFooter([]);
        $mpdf->SetDisplayMode('fullpage');

        $mpdf->WriteHTML($html);

        return $mpdf->Output('', Destination::STRING_RETURN);
    }

    protected function formatDateTime(?string $occurredAt): string
    {
        if (! $occurredAt) {
            return now()->format('jS M Y g:i A');
        }

        try {
            return Carbon::parse($occurredAt)->format('jS M Y g:i A');
        } catch (Throwable) {
            return $occurredAt;
        }
    }

    protected function maskPhoneNumber(?string $phoneOrBankAccountNumber): string
    {
        if (! is_string($phoneOrBankAccountNumber) || $phoneOrBankAccountNumber === '') {
            return 'N/A';
        }

        $digits = preg_replace('/\D+/', '', $phoneOrBankAccountNumber);
        if (! is_string($digits) || strlen($digits) < 7) {
            return $phoneOrBankAccountNumber;
        }

        return substr($digits, 0, 4).'***'.substr($digits, -3);
    }
}
