<?php

namespace App\Services\Sms;

use App\Services\Trading\TradingSettingsService;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Throwable;

class InfobipSms implements SmsProvider
{
    protected string $baseUrl;

    protected string $apiKey;

    protected ?string $from;

    public function __construct()
    {
        $this->baseUrl = config('services.infobip.base_url');
        $this->apiKey = config('services.infobip.api_key');
        $this->from = config('services.infobip.from');
    }

    public function send(?string $phoneNumber, string $message): bool
    {
        if (! $this->smsNotificationsEnabled() || ! $phoneNumber || ! $this->baseUrl || ! $this->apiKey) {
            Log::warning('Infobip SMS skip: missing phone number or configuration.');

            return false;
        }

        $phoneNumber = $this->normalizePhoneNumber($phoneNumber);

        try {
            $response = Http::withHeaders([
                'Authorization' => "App {$this->apiKey}",
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ])->post("{$this->baseUrl}/sms/2/text/advanced", [
                'messages' => [
                    [
                        'from' => $this->from ?: 'Service',
                        'destinations' => [
                            ['to' => $phoneNumber],
                        ],
                        'text' => $message,
                    ],
                ],
            ]);

            if ($response->successful()) {
                Log::info("Infobip SMS sent successfully to {$phoneNumber}");

                return true;
            }

            Log::error('Infobip SMS failed.', [
                'status' => $response->status(),
                'response' => $response->json(),
                'phone' => $phoneNumber,
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error('Infobip SMS exception: '.$e->getMessage());

            return false;
        }
    }

    protected function normalizePhoneNumber(string $phoneNumber): string
    {
        // Remove non-numeric characters except +
        $phone = preg_replace('/[^\d+]/', '', $phoneNumber);

        // If it starts with 0 and matches 10 digits (Kenyan format), replace with +254
        if (preg_match('/^0(7|1)\d{8}$/', $phone)) {
            $phone = '+254'.substr($phone, 1);
        }

        // Ensure it starts with +
        if (! str_starts_with($phone, '+')) {
            $phone = '+'.$phone;
        }

        return $phone;
    }

    protected function smsNotificationsEnabled(): bool
    {
        try {
            return app(TradingSettingsService::class)->smsNotificationsEnabled();
        } catch (Throwable) {
            return true;
        }
    }
}
