<?php

namespace App\Services\Sms;

use App\Models\User;
use App\Services\Trading\TradingSettingsService;
use Throwable;

class SmsNotifier
{
    public function __construct(
        protected SmsProvider $smsProvider,
    ) {}

    public function sendToPhone(?string $phoneNumber, string $message, ?string $type = null): void
    {
        $enabled = $type !== null
            ? $this->isSmsTypeEnabled($type)
            : $this->smsNotificationsEnabled();

        if (! $enabled || ! filled($phoneNumber) || ! filled($message)) {
            return;
        }

        $this->smsProvider->send((string) $phoneNumber, $message);
    }

    public function sendToAdmins(string $message, ?string $type = null): void
    {
        $enabled = $type !== null
            ? $this->isSmsTypeEnabled($type)
            : $this->smsNotificationsEnabled();

        if (! $enabled || ! filled($message)) {
            return;
        }

        $sentPhones = [];
        $adminPhones = config('services.admin_phones', []);
        if (! is_array($adminPhones)) {
            $adminPhones = [];
        }

        foreach ($adminPhones as $adminPhone) {
            if (! is_string($adminPhone) || trim($adminPhone) === '') {
                continue;
            }

            $normalizedPhone = trim($adminPhone);
            $this->smsProvider->send($normalizedPhone, $message);
            $sentPhones[] = $normalizedPhone;
        }

        $adminEmail = (string) config('mail.admin_address', '');
        if (! filled($adminEmail)) {
            return;
        }

        $adminUserPhone = User::query()
            ->where('email', $adminEmail)
            ->value('phone');

        if (! is_string($adminUserPhone) || trim($adminUserPhone) === '') {
            return;
        }

        $adminUserPhone = trim($adminUserPhone);
        if (in_array($adminUserPhone, $sentPhones, true)) {
            return;
        }

        $this->smsProvider->send($adminUserPhone, $message);
    }

    protected function smsNotificationsEnabled(): bool
    {
        try {
            return app(TradingSettingsService::class)->smsNotificationsEnabled();
        } catch (Throwable) {
            return true;
        }
    }

    protected function isSmsTypeEnabled(string $type): bool
    {
        try {
            return app(TradingSettingsService::class)->isSmsTypeEnabled($type);
        } catch (Throwable) {
            return true;
        }
    }
}
