<?php

namespace App\Services\Trading;

use App\Models\SecurityOrder;
use App\Services\Vrm\UtilityService;

class TradingSettingsService
{
    private const string SETTINGS_TYPE = 'general';

    private const string KEY_MARKET_OPEN = 'market_open';

    private const string KEY_INVESTMENT_DURATION_DAYS = 'investment_duration_days';

    private const string KEY_INVESTMENT_RETURN_PERCENT = 'investment_return_percent';

    private const string KEY_WITHDRAWAL_MIN_AMOUNT = 'withdrawal_min_amount'; // legacy fallback

    private const string KEY_WITHDRAWAL_MAX_AMOUNT = 'withdrawal_max_amount'; // legacy fallback

    private const string KEY_WITHDRAWAL_MPESA_MIN_AMOUNT = 'withdrawal_mpesa_min_amount';

    private const string KEY_WITHDRAWAL_MPESA_MAX_AMOUNT = 'withdrawal_mpesa_max_amount';

    private const string KEY_WITHDRAWAL_BANK_MIN_AMOUNT = 'withdrawal_bank_min_amount';

    private const string KEY_WITHDRAWAL_BANK_MAX_AMOUNT = 'withdrawal_bank_max_amount';

    private const string KEY_SELL_MIN_SHARES_PER_REQUEST = 'sell_min_shares_per_request';

    private const string KEY_SELL_MAX_SHARES_PER_REQUEST = 'sell_max_shares_per_request';

    private const string KEY_SELL_MAX_SHARES_PER_DAY = 'sell_max_shares_per_day';

    private const string KEY_DEPOSIT_MIN_AMOUNT = 'deposit_min_amount';

    private const string KEY_DEPOSIT_MAX_AMOUNT = 'deposit_max_amount';

    private const string KEY_NOTIFICATION_EMAIL_ENABLED = 'notification_email_enabled';

    private const string KEY_NOTIFICATION_SMS_ENABLED = 'notification_sms_enabled';

    private const string KEY_NOTIFICATION_USER_OTP_SMS_ENABLED = 'notification_user_otp_sms_enabled';

    private const string KEY_NOTIFICATION_USER_OTP_EMAIL_ENABLED = 'notification_user_otp_email_enabled';

    private const string KEY_NOTIFICATION_USER_TRANSACTION_SMS_ENABLED = 'notification_user_transaction_sms_enabled';

    private const string KEY_NOTIFICATION_USER_TRANSACTION_EMAIL_ENABLED = 'notification_user_transaction_email_enabled';

    private const string KEY_NOTIFICATION_USER_WELCOME_SMS_ENABLED = 'notification_user_welcome_sms_enabled';

    private const string KEY_NOTIFICATION_USER_WELCOME_EMAIL_ENABLED = 'notification_user_welcome_email_enabled';

    private const string KEY_NOTIFICATION_USER_PASSWORD_UPDATED_SMS_ENABLED = 'notification_user_password_updated_sms_enabled';

    private const string KEY_NOTIFICATION_USER_PASSWORD_UPDATED_EMAIL_ENABLED = 'notification_user_password_updated_email_enabled';

    private const string KEY_NOTIFICATION_ADMIN_DEPOSIT_SMS_ENABLED = 'notification_admin_deposit_sms_enabled';

    private const string KEY_NOTIFICATION_ADMIN_DEPOSIT_EMAIL_ENABLED = 'notification_admin_deposit_email_enabled';

    private const string KEY_NOTIFICATION_ADMIN_TRANSACTION_SMS_ENABLED = 'notification_admin_transaction_sms_enabled';

    private const string KEY_NOTIFICATION_ADMIN_TRANSACTION_EMAIL_ENABLED = 'notification_admin_transaction_email_enabled';

    public function __construct(
        private UtilityService $utilityService,
    ) {}

    public function isMarketOpen(): bool
    {
        return (bool) (int) $this->utilityService->get(self::KEY_MARKET_OPEN, '1', self::SETTINGS_TYPE);
    }

    public function investmentDurationDays(): int
    {
        $days = (int) $this->utilityService->get(self::KEY_INVESTMENT_DURATION_DAYS, '30', self::SETTINGS_TYPE);

        return max(1, $days);
    }

    public function investmentReturnPercent(): float
    {
        $percent = (float) $this->utilityService->get(self::KEY_INVESTMENT_RETURN_PERCENT, '100', self::SETTINGS_TYPE);

        return max(0, $percent);
    }

    public function calculateExpectedReturn(float $principalAmount): float
    {
        $returnPercent = $this->investmentReturnPercent();

        return $principalAmount * (1 + ($returnPercent / 100));
    }

    public function withdrawalMinAmount(string $channel = 'mpesa'): float
    {
        [$minKey] = $this->withdrawalKeysForChannel($channel);
        $raw = $this->utilityService->get($minKey, '', self::SETTINGS_TYPE);
        $fallback = $this->utilityService->get(self::KEY_WITHDRAWAL_MIN_AMOUNT, '0', self::SETTINGS_TYPE);
        $min = (float) ($raw === '' ? $fallback : $raw);

        return max(0, $min);
    }

    public function withdrawalMaxAmount(string $channel = 'mpesa'): float
    {
        [, $maxKey] = $this->withdrawalKeysForChannel($channel);
        $raw = $this->utilityService->get($maxKey, '', self::SETTINGS_TYPE);
        $fallback = $this->utilityService->get(self::KEY_WITHDRAWAL_MAX_AMOUNT, '0', self::SETTINGS_TYPE);
        $max = (float) ($raw === '' ? $fallback : $raw);

        return $max > 0 ? $max : PHP_FLOAT_MAX;
    }

    public function withdrawalLimits(string $channel = 'mpesa'): array
    {
        return [
            'min' => $this->withdrawalMinAmount($channel),
            'max' => $this->withdrawalMaxAmount($channel),
        ];
    }

    public function sellMinSharesPerRequest(): int
    {
        $min = (int) $this->utilityService->get(self::KEY_SELL_MIN_SHARES_PER_REQUEST, '1', self::SETTINGS_TYPE);

        return max(1, $min);
    }

    public function sellMaxSharesPerRequest(): int
    {
        $max = (int) $this->utilityService->get(self::KEY_SELL_MAX_SHARES_PER_REQUEST, '0', self::SETTINGS_TYPE);

        return max(0, $max);
    }

    public function sellMaxSharesPerDay(): int
    {
        $max = (int) $this->utilityService->get(self::KEY_SELL_MAX_SHARES_PER_DAY, '0', self::SETTINGS_TYPE);

        return max(0, $max);
    }

    public function sellLimits(): array
    {
        return [
            'min_per_request' => $this->sellMinSharesPerRequest(),
            'max_per_request' => $this->sellMaxSharesPerRequest(),
            'max_per_day' => $this->sellMaxSharesPerDay(),
        ];
    }

    public function depositMinAmount(): float
    {
        $min = (float) $this->utilityService->get(self::KEY_DEPOSIT_MIN_AMOUNT, '100', self::SETTINGS_TYPE);

        return max(0, $min);
    }

    public function depositMaxAmount(): float
    {
        $max = (float) $this->utilityService->get(self::KEY_DEPOSIT_MAX_AMOUNT, '0', self::SETTINGS_TYPE);

        return $max > 0 ? $max : PHP_FLOAT_MAX;
    }

    public function depositLimits(): array
    {
        return [
            'min' => $this->depositMinAmount(),
            'max' => $this->depositMaxAmount(),
        ];
    }

    public function emailNotificationsEnabled(): bool
    {
        return (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_EMAIL_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function smsNotificationsEnabled(): bool
    {
        return (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_SMS_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function userOtpSmsEnabled(): bool
    {
        return $this->smsNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_USER_OTP_SMS_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function userOtpEmailEnabled(): bool
    {
        return $this->emailNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_USER_OTP_EMAIL_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function userTransactionSmsEnabled(): bool
    {
        return $this->smsNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_USER_TRANSACTION_SMS_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function userTransactionEmailEnabled(): bool
    {
        return $this->emailNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_USER_TRANSACTION_EMAIL_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function userWelcomeSmsEnabled(): bool
    {
        return $this->smsNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_USER_WELCOME_SMS_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function userWelcomeEmailEnabled(): bool
    {
        return $this->emailNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_USER_WELCOME_EMAIL_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function userPasswordUpdatedSmsEnabled(): bool
    {
        return $this->smsNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_USER_PASSWORD_UPDATED_SMS_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function userPasswordUpdatedEmailEnabled(): bool
    {
        return $this->emailNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_USER_PASSWORD_UPDATED_EMAIL_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function adminDepositSmsEnabled(): bool
    {
        return $this->smsNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_ADMIN_DEPOSIT_SMS_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function adminDepositEmailEnabled(): bool
    {
        return $this->emailNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_ADMIN_DEPOSIT_EMAIL_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function adminTransactionSmsEnabled(): bool
    {
        return $this->smsNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_ADMIN_TRANSACTION_SMS_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function adminTransactionEmailEnabled(): bool
    {
        return $this->emailNotificationsEnabled()
            && (bool) (int) $this->utilityService->get(self::KEY_NOTIFICATION_ADMIN_TRANSACTION_EMAIL_ENABLED, '1', self::SETTINGS_TYPE);
    }

    public function isSmsTypeEnabled(string $type): bool
    {
        return match ($type) {
            'user_otp' => $this->userOtpSmsEnabled(),
            'user_transaction' => $this->userTransactionSmsEnabled(),
            'user_welcome' => $this->userWelcomeSmsEnabled(),
            'user_password_updated' => $this->userPasswordUpdatedSmsEnabled(),
            'admin_deposit' => $this->adminDepositSmsEnabled(),
            'admin_transaction' => $this->adminTransactionSmsEnabled(),
            default => $this->smsNotificationsEnabled(),
        };
    }

    public function isEmailTypeEnabled(string $type): bool
    {
        return match ($type) {
            'user_otp' => $this->userOtpEmailEnabled(),
            'user_transaction' => $this->userTransactionEmailEnabled(),
            'user_welcome' => $this->userWelcomeEmailEnabled(),
            'user_password_updated' => $this->userPasswordUpdatedEmailEnabled(),
            'admin_deposit' => $this->adminDepositEmailEnabled(),
            'admin_transaction' => $this->adminTransactionEmailEnabled(),
            default => $this->emailNotificationsEnabled(),
        };
    }

    /**
     * Get raw granular notification value (for admin UI) - not ANDed with global.
     */
    public function getGranularNotificationRaw(string $key): bool
    {
        return (bool) (int) $this->utilityService->get($key, '1', self::SETTINGS_TYPE);
    }

    public function soldSharesTodayByUser(int $userId): int
    {
        return (int) SecurityOrder::query()
            ->where('user_id', $userId)
            ->where('order_type', 'sell')
            ->whereDate('traded_at', today())
            ->whereIn('status', ['pending', 'approved', 'completed'])
            ->sum('quantity');
    }

    public function remainingDailySellSharesForUser(int $userId): int
    {
        $dailyMax = $this->sellMaxSharesPerDay();
        if ($dailyMax <= 0) {
            return PHP_INT_MAX;
        }

        return max(0, $dailyMax - $this->soldSharesTodayByUser($userId));
    }

    public function updateSettings(bool $marketOpen, int $investmentDurationDays, float $investmentReturnPercent): void
    {
        $this->utilityService->setMultiple([
            self::KEY_MARKET_OPEN => $marketOpen ? '1' : '0',
            self::KEY_INVESTMENT_DURATION_DAYS => (string) max(1, $investmentDurationDays),
            self::KEY_INVESTMENT_RETURN_PERCENT => (string) max(0, $investmentReturnPercent),
        ], self::SETTINGS_TYPE);
    }

    public function updateWithdrawalLimits(float $mpesaMinAmount, float $mpesaMaxAmount, float $bankMinAmount, float $bankMaxAmount): void
    {
        $this->utilityService->setMultiple([
            self::KEY_WITHDRAWAL_MPESA_MIN_AMOUNT => (string) max(0, $mpesaMinAmount),
            self::KEY_WITHDRAWAL_MPESA_MAX_AMOUNT => (string) max(0, $mpesaMaxAmount),
            self::KEY_WITHDRAWAL_BANK_MIN_AMOUNT => (string) max(0, $bankMinAmount),
            self::KEY_WITHDRAWAL_BANK_MAX_AMOUNT => (string) max(0, $bankMaxAmount),
        ], self::SETTINGS_TYPE);
    }

    public function updateSellLimits(int $minPerRequest, int $maxPerRequest, int $maxPerDay): void
    {
        $this->utilityService->setMultiple([
            self::KEY_SELL_MIN_SHARES_PER_REQUEST => (string) max(1, $minPerRequest),
            self::KEY_SELL_MAX_SHARES_PER_REQUEST => (string) max(0, $maxPerRequest),
            self::KEY_SELL_MAX_SHARES_PER_DAY => (string) max(0, $maxPerDay),
        ], self::SETTINGS_TYPE);
    }

    public function updateDepositLimits(float $minAmount, float $maxAmount): void
    {
        $this->utilityService->setMultiple([
            self::KEY_DEPOSIT_MIN_AMOUNT => (string) max(0, $minAmount),
            self::KEY_DEPOSIT_MAX_AMOUNT => (string) max(0, $maxAmount),
        ], self::SETTINGS_TYPE);
    }

    public function updateNotificationChannels(bool $emailEnabled, bool $smsEnabled): void
    {
        $this->utilityService->setMultiple([
            self::KEY_NOTIFICATION_EMAIL_ENABLED => $emailEnabled ? '1' : '0',
            self::KEY_NOTIFICATION_SMS_ENABLED => $smsEnabled ? '1' : '0',
        ], self::SETTINGS_TYPE);
    }

    /**
     * @param  array<string, bool>  $settings
     */
    public function updateNotificationChannelsGranular(array $settings): void
    {
        $keys = [
            'notification_user_otp_sms_enabled' => self::KEY_NOTIFICATION_USER_OTP_SMS_ENABLED,
            'notification_user_otp_email_enabled' => self::KEY_NOTIFICATION_USER_OTP_EMAIL_ENABLED,
            'notification_user_transaction_sms_enabled' => self::KEY_NOTIFICATION_USER_TRANSACTION_SMS_ENABLED,
            'notification_user_transaction_email_enabled' => self::KEY_NOTIFICATION_USER_TRANSACTION_EMAIL_ENABLED,
            'notification_user_welcome_sms_enabled' => self::KEY_NOTIFICATION_USER_WELCOME_SMS_ENABLED,
            'notification_user_welcome_email_enabled' => self::KEY_NOTIFICATION_USER_WELCOME_EMAIL_ENABLED,
            'notification_user_password_updated_sms_enabled' => self::KEY_NOTIFICATION_USER_PASSWORD_UPDATED_SMS_ENABLED,
            'notification_user_password_updated_email_enabled' => self::KEY_NOTIFICATION_USER_PASSWORD_UPDATED_EMAIL_ENABLED,
            'notification_admin_deposit_sms_enabled' => self::KEY_NOTIFICATION_ADMIN_DEPOSIT_SMS_ENABLED,
            'notification_admin_deposit_email_enabled' => self::KEY_NOTIFICATION_ADMIN_DEPOSIT_EMAIL_ENABLED,
            'notification_admin_transaction_sms_enabled' => self::KEY_NOTIFICATION_ADMIN_TRANSACTION_SMS_ENABLED,
            'notification_admin_transaction_email_enabled' => self::KEY_NOTIFICATION_ADMIN_TRANSACTION_EMAIL_ENABLED,
        ];

        $values = [];
        foreach ($keys as $inputKey => $storageKey) {
            $values[$storageKey] = ($settings[$inputKey] ?? true) ? '1' : '0';
        }

        $this->utilityService->setMultiple($values, self::SETTINGS_TYPE);
    }

    private function withdrawalKeysForChannel(string $channel): array
    {
        return $channel === 'bank'
            ? [self::KEY_WITHDRAWAL_BANK_MIN_AMOUNT, self::KEY_WITHDRAWAL_BANK_MAX_AMOUNT]
            : [self::KEY_WITHDRAWAL_MPESA_MIN_AMOUNT, self::KEY_WITHDRAWAL_MPESA_MAX_AMOUNT];
    }
}
