<?php

namespace App\Services;

use App\Models\Security;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Http\Client\RequestException;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Throwable;

class YahooNSE implements NSEService
{
    /**
     * @var array<int, string>
     */
    private const DEFAULT_SYMBOLS = [
        'ABSA',
        'AMAC',
        'BAT',
        'BKG',
        'BRIT',
        'CARB',
        'CIC',
        'COOP',
        'CRWN',
        'DTK',
        'EABL',
        'EQTY',
        'HFCK',
        'SCOM',
        'SCBK',
        'SBIC',
        'NSE',
        'NCBA',
        'KPLC',
        'KQ',
        'KCB',
        'JUB',
        'IMH',
        'KEGN',
        'TOTL',
        'XPRS',
    ];

    public function __construct(
        private readonly YahooNSEOutputHelper $outputHelper,
        private readonly RapidNSE $rapidNSE
    ) {}

    /**
     * @return array<int|string, mixed>
     */
    public function getStocks(): array
    {
        $symbols = $this->configuredSymbols();
        if (! $this->isConfigured() || $symbols === []) {
            return $this->rapidFallbackStocks();
        }

        try {
            $response = Http::connectTimeout(3)
                ->timeout($this->timeout())
                ->acceptJson()
                ->withHeaders($this->defaultHeaders())
                ->get($this->baseUrl().'/v7/finance/quote', [
                    'symbols' => implode(',', $symbols),
                ]);

            if (! $response->successful()) {
                Log::warning('YahooNSE API request failed', [
                    'endpoint' => 'quote',
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);
                $response->throw();
            }

            $quotes = $response->json('quoteResponse.result', []);
            if (config('app.debug')) {
                Log::debug('YahooNSE raw quote payload', [
                    'symbols' => $symbols,
                    'count' => is_array($quotes) ? count($quotes) : 0,
                    'first' => is_array($quotes) ? ($quotes[0] ?? null) : null,
                ]);
            }

            if (! is_array($quotes)) {
                return $this->getStocksFromCharts($symbols);
            }

            $formatted = $this->outputHelper->formatStocks($quotes);
            if (($formatted['data']['data'] ?? []) === []) {
                return $this->ensureCompleteStocksPayload($this->getStocksFromCharts($symbols), $symbols);
            }

            if (config('app.debug')) {
                Log::debug('YahooNSE formatted stocks payload', [
                    'count' => count($formatted['data']['data'] ?? []),
                    'first' => $formatted['data']['data'][0] ?? null,
                ]);
            }

            return $this->ensureCompleteStocksPayload($formatted, $symbols);
        } catch (ConnectionException|RequestException|Throwable $e) {
            Log::warning('YahooNSE stocks request failed', [
                'endpoint' => 'quote',
                'message' => $e->getMessage(),
            ]);

            $fallback = $this->getStocksFromCharts($symbols);

            if (($fallback['data']['data'] ?? []) !== []) {
                return $this->ensureCompleteStocksPayload($fallback, $symbols);
            }

            return $this->ensureCompleteStocksPayload($this->rapidFallbackStocks(), $symbols);
        }
    }

    /**
     * @return array<int|string, mixed>
     */
    public function getStock(string $symbol): array
    {
        if (! $this->isConfigured()) {
            return $this->rapidFallbackStock($symbol);
        }

        $normalizedSymbol = $this->toYahooSymbol($symbol);
        if ($normalizedSymbol === '') {
            return $this->rapidFallbackStock($symbol);
        }

        try {
            $response = Http::connectTimeout(3)
                ->timeout($this->timeout())
                ->acceptJson()
                ->withHeaders($this->defaultHeaders())
                ->get($this->baseUrl().'/v8/finance/chart/'.urlencode($normalizedSymbol), [
                    'interval' => '1d',
                    'range' => '5d',
                ]);

            if (! $response->successful()) {
                Log::warning('YahooNSE API request failed', [
                    'symbol' => $symbol,
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);
                $response->throw();
            }

            $result = $response->json('chart.result.0', []);
            if (config('app.debug')) {
                Log::debug('YahooNSE raw chart payload', [
                    'symbol' => $normalizedSymbol,
                    'meta' => is_array($result) ? ($result['meta'] ?? null) : null,
                ]);
            }

            if (! is_array($result) || $result === []) {
                return $this->rapidFallbackStock($symbol);
            }

            $formatted = $this->outputHelper->formatStock($normalizedSymbol, $result);
            if (config('app.debug')) {
                Log::debug('YahooNSE formatted stock payload', [
                    'symbol' => $normalizedSymbol,
                    'first' => $formatted['data']['data'][0] ?? null,
                ]);
            }

            return $formatted;
        } catch (ConnectionException|RequestException|Throwable $e) {
            Log::warning('YahooNSE stock request failed', [
                'symbol' => $symbol,
                'message' => $e->getMessage(),
            ]);

            return $this->rapidFallbackStock($symbol);
        }
    }

    private function baseUrl(): string
    {
        $configuredBaseUrl = trim((string) config('services.yahoo_nse.base_url', ''));

        return rtrim($configuredBaseUrl !== '' ? $configuredBaseUrl : 'https://query1.finance.yahoo.com', '/');
    }

    /**
     * @return array<int, string>
     */
    private function configuredSymbols(): array
    {
        $symbols = config('services.yahoo_nse.symbols', []);
        if (! is_array($symbols) || $symbols === []) {
            $symbols = self::DEFAULT_SYMBOLS;
        }

        $normalized = array_values(array_filter(array_map(function (mixed $symbol): string {
            return $this->toYahooSymbol((string) $symbol);
        }, $symbols)));

        if ($normalized === []) {
            return array_map(fn (string $symbol): string => $this->toYahooSymbol($symbol), self::DEFAULT_SYMBOLS);
        }

        return $normalized;
    }

    private function timeout(): int
    {
        $configuredTimeout = (int) config('services.yahoo_nse.timeout', 8);

        return $configuredTimeout > 0 ? $configuredTimeout : 8;
    }

    private function isConfigured(): bool
    {
        return $this->baseUrl() !== '';
    }

    private function toYahooSymbol(string $symbol): string
    {
        $cleaned = strtoupper(trim($symbol));
        if ($cleaned === '') {
            return '';
        }

        if (! str_contains($cleaned, '.')) {
            $cleaned .= '.NR';
        }

        return $cleaned;
    }

    /**
     * @return array<string, string>
     */
    private function defaultHeaders(): array
    {
        return [
            'User-Agent' => 'Mozilla/5.0 (compatible; FxZidii/1.0)',
            'Accept' => 'application/json',
        ];
    }

    /**
     * @param  array<int, string>  $symbols
     * @return array<int|string, mixed>
     */
    private function getStocksFromCharts(array $symbols): array
    {
        $rows = [];

        foreach ($symbols as $symbol) {
            try {
                $response = Http::connectTimeout(3)
                    ->timeout($this->timeout())
                    ->acceptJson()
                    ->withHeaders($this->defaultHeaders())
                    ->get($this->baseUrl().'/v8/finance/chart/'.urlencode($symbol), [
                        'interval' => '1d',
                        'range' => '5d',
                    ]);

                if (! $response->successful()) {
                    continue;
                }

                $result = $response->json('chart.result.0', []);
                if (! is_array($result) || $result === []) {
                    continue;
                }

                $formatted = $this->outputHelper->formatStock($symbol, $result);
                $row = $formatted['data']['data'][0] ?? null;

                if (is_array($row) && ($row['ticker'] ?? '') !== '') {
                    $rows[] = $row;
                }
            } catch (Throwable) {
                continue;
            }
        }

        if (config('app.debug')) {
            Log::debug('YahooNSE chart fallback payload', [
                'symbols' => $symbols,
                'count' => count($rows),
                'first' => $rows[0] ?? null,
            ]);
        }

        return $this->outputHelper->formatRows($rows, 'Stocks list retrieved.');
    }

    /**
     * @return array<int|string, mixed>
     */
    private function rapidFallbackStocks(): array
    {
        $rapid = $this->rapidNSE->getStocks();

        if ($rapid !== []) {
            return $rapid;
        }

        return $this->databaseFallbackStocks();
    }

    /**
     * @return array<int|string, mixed>
     */
    private function rapidFallbackStock(string $symbol): array
    {
        $rapid = $this->rapidNSE->getStock($symbol);

        if ($rapid !== []) {
            return $rapid;
        }

        return $this->databaseFallbackStock($symbol);
    }

    /**
     * @return array<int|string, mixed>
     */
    private function databaseFallbackStocks(): array
    {
        return $this->outputHelper->formatRows(
            $this->baseRowsForConfiguredSymbols($this->configuredSymbols()),
            'Stocks list retrieved.'
        );
    }

    /**
     * @return array<int|string, mixed>
     */
    private function databaseFallbackStock(string $symbol): array
    {
        $normalizedSymbol = strtoupper(trim($symbol));
        if ($normalizedSymbol === '') {
            return [];
        }

        $security = Security::query()
            ->where('trading_name', $normalizedSymbol)
            ->where('is_active', true)
            ->with('latestLog')
            ->first();

        if (! $security instanceof Security) {
            return $this->outputHelper->formatRows([[
                'ticker' => $normalizedSymbol,
                'name' => $normalizedSymbol,
                'volume' => '0',
                'price' => '0.00',
                'change' => '+0.00',
            ]], 'Stock data retrieved.');
        }

        return $this->outputHelper->formatRows([[
            'ticker' => $normalizedSymbol,
            'name' => (string) ($security->name ?: $normalizedSymbol),
            'volume' => '0',
            'price' => number_format($security->latestPrice(), 2, '.', ''),
            'change' => $this->formatSignedChange($security->latestChangePercent()),
        ]], 'Stock data retrieved.');
    }

    /**
     * @param  array<int|string, mixed>  $payload
     * @param  array<int, string>  $symbols
     * @return array<int|string, mixed>
     */
    private function ensureCompleteStocksPayload(array $payload, array $symbols): array
    {
        $baseRows = $this->baseRowsForConfiguredSymbols($symbols);
        $liveRows = $this->extractRows($payload);
        $liveByTicker = [];

        foreach ($liveRows as $row) {
            if (! is_array($row)) {
                continue;
            }

            $ticker = strtoupper(trim((string) ($row['ticker'] ?? $row['symbol'] ?? '')));
            if ($ticker === '') {
                continue;
            }

            $liveByTicker[$ticker] = [
                'ticker' => $ticker,
                'name' => (string) ($row['name'] ?? $ticker),
                'volume' => (string) ($row['volume'] ?? '0'),
                'price' => is_string($row['price'] ?? null) ? $row['price'] : number_format((float) ($row['price'] ?? 0), 2, '.', ''),
                'change' => (string) ($row['change'] ?? '+0.00'),
            ];
        }

        $finalRows = [];
        foreach ($baseRows as $baseRow) {
            $ticker = $baseRow['ticker'];
            $finalRows[] = $liveByTicker[$ticker] ?? $baseRow;
        }

        return $this->outputHelper->formatRows($finalRows, 'Stocks list retrieved.');
    }

    /**
     * @param  array<int, string>  $symbols
     * @return array<int, array{ticker: string, name: string, volume: string, price: string, change: string}>
     */
    private function baseRowsForConfiguredSymbols(array $symbols): array
    {
        $tickers = array_values(array_unique(array_map(fn (string $symbol): string => strtoupper((string) preg_replace('/\.NR$/i', '', $symbol)), $symbols)));

        $securities = Security::withTrashed()
            ->whereIn('trading_name', $tickers)
            ->with('latestLog')
            ->get()
            ->keyBy(fn (Security $security): string => strtoupper(trim((string) $security->trading_name)));

        return array_map(function (string $ticker) use ($securities): array {
            $security = $securities->get($ticker);

            return [
                'ticker' => $ticker,
                'name' => (string) ($security?->name ?: $ticker),
                'volume' => '0',
                'price' => number_format($security?->latestPrice() ?? 0, 2, '.', ''),
                'change' => $this->formatSignedChange($security?->latestChangePercent() ?? 0),
            ];
        }, $tickers);
    }

    private function formatSignedChange(float $value): string
    {
        $prefix = $value >= 0 ? '+' : '';

        return $prefix.number_format($value, 2, '.', '');
    }

    /**
     * @param  array<int|string, mixed>  $payload
     * @return array<int, mixed>
     */
    private function extractRows(array $payload): array
    {
        $rows = $payload['data']['data'] ?? $payload['data'] ?? $payload;

        return is_array($rows) ? array_values($rows) : [];
    }
}
