<?php

use App\Models\CommunityPost;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Livewire\WithPagination;

new class extends Component {
    use WithNotifications;
    use WithPagination;

    public string $search = '';

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    #[Computed]
    public function posts()
    {
        return CommunityPost::query()
            ->with('author:id,name')
            ->when($this->search !== '', function ($query): void {
                $query->where('title', 'like', "%{$this->search}%");
            })
            ->orderByDesc('created_at')
            ->paginate(10);
    }

    public function publishNow(int $postId): void
    {
        $post = CommunityPost::query()->find($postId);

        if (! $post) {
            $this->notifyError(__('Community post not found.'));

            return;
        }

        $post->update([
            'is_active' => true,
            'published_at' => now(),
        ]);

        $this->notifySuccess(__('Community post published.'));
    }

    public function unpublish(int $postId): void
    {
        $post = CommunityPost::query()->find($postId);

        if (! $post) {
            $this->notifyError(__('Community post not found.'));

            return;
        }

        $post->update([
            'is_active' => false,
        ]);

        $this->notifyInfo(__('Community post disabled.'));
    }

    public function delete(int $postId): void
    {
        $post = CommunityPost::query()->find($postId);

        if (! $post) {
            $this->notifyError(__('Community post not found.'));

            return;
        }

        $post->delete();

        $this->notifySuccess(__('Community post deleted.'));
    }
}; ?>

<div>
    <x-admin-panel>
        <x-slot name="header">{{ __('Community Center') }}</x-slot>
        <x-slot name="desc">{{ __('Create and manage notice board posts for members.') }}</x-slot>
        <x-slot name="button">
            <a href="{{ route('admin.community.create') }}" wire:navigate
                class="rounded-md bg-blue-600 px-3 py-2 text-sm font-bold text-white hover:bg-blue-500">
                {{ __('Create Post') }}
            </a>
        </x-slot>

        <div class="mb-4 mt-2">
            <input type="text" wire:model.live.debounce.300ms="search"
                class="block w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 outline-none focus:border-indigo-500 dark:border-gray-600 dark:bg-gray-700 dark:text-gray-100"
                placeholder="{{ __('Search by title...') }}" />
        </div>

        {!! $this->renderNotification() !!}

        <div class="overflow-hidden rounded-lg shadow-sm ring-1 ring-black/5 dark:ring-white/10">
            <table class="min-w-full divide-y divide-gray-300 dark:divide-gray-600">
                <thead class="bg-gray-50 dark:bg-gray-700">
                    <tr>
                        <th class="px-4 py-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Title') }}</th>
                        <th class="px-4 py-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Author') }}</th>
                        <th class="px-4 py-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Publication Date') }}</th>
                        <th class="px-4 py-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Status') }}</th>
                        <th class="px-4 py-3 text-right text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Actions') }}</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-gray-100 bg-white dark:divide-gray-700 dark:bg-gray-800">
                    @forelse ($this->posts as $post)
                        <tr>
                            <td class="px-4 py-4 text-sm text-gray-900 dark:text-gray-100">{{ $post->title }}</td>
                            <td class="px-4 py-4 text-sm text-gray-900 dark:text-gray-100">{{ $post->author?->name ?? '-' }}</td>
                            <td class="px-4 py-4 text-sm text-gray-900 dark:text-gray-100">{{ optional($post->published_at)->format('Y-m-d H:i') ?? '-' }}</td>
                            <td class="px-4 py-4 text-sm">
                                @if ($post->is_active && $post->published_at && $post->published_at->lte(now()))
                                    <span class="rounded bg-emerald-500 px-2 py-0.5 text-xs font-semibold text-white">{{ __('Published') }}</span>
                                @elseif ($post->is_active && $post->published_at && $post->published_at->isFuture())
                                    <span class="rounded bg-amber-500 px-2 py-0.5 text-xs font-semibold text-white">{{ __('Scheduled') }}</span>
                                @else
                                    <span class="rounded bg-gray-500 px-2 py-0.5 text-xs font-semibold text-white">{{ __('Disabled') }}</span>
                                @endif
                            </td>
                            <td class="px-4 py-4 text-right text-xs">
                                <a href="{{ route('admin.community.edit', ['post' => $post->id]) }}" wire:navigate
                                    class="inline-flex items-center rounded-md bg-blue-600 px-2.5 py-1 font-semibold text-white hover:bg-blue-500">{{ __('Edit') }}</a>

                                @if ($post->is_active)
                                    <button type="button" wire:click="unpublish({{ $post->id }})"
                                        class="inline-flex items-center rounded-md bg-yellow-500 px-2.5 py-1 font-semibold text-white hover:bg-yellow-400">{{ __('Disable') }}</button>
                                @else
                                    <button type="button" wire:click="publishNow({{ $post->id }})"
                                        class="inline-flex items-center rounded-md bg-emerald-600 px-2.5 py-1 font-semibold text-white hover:bg-emerald-500">{{ __('Enable') }}</button>
                                @endif

                                <button type="button" wire:click="$js.confirmDelete({{ $post->id }})"
                                    class="inline-flex items-center rounded-md bg-red-600 px-2.5 py-1 font-semibold text-white hover:bg-red-500">{{ __('Delete') }}</button>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="5" class="px-4 py-6 text-center text-sm text-gray-500 dark:text-gray-400">{{ __('No community posts found.') }}</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <div class="mt-4">{{ $this->posts->links() }}</div>
    </x-admin-panel>

    @script
        <script>
            $js('confirmDelete', async (postId) => {
                const result = await Swal.fire({
                    title: 'Delete post?',
                    icon: 'warning',
                    showCancelButton: true,
                });

                if (result.isConfirmed) {
                    $wire.delete(postId);
                }
            });
        </script>
    @endscript
</div>
