<?php

use App\Facades\Vrm\MediaForge;
use App\Models\Security;
use App\Models\SecurityLog;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

new class extends Component {
    use WithFileUploads;
    use WithNotifications;

    public ?Security $security = null;

    #[Validate('required|string|max:20|unique:securities,trading_name')]
    public string $trading_name = '';

    #[Validate('nullable|string|max:255')]
    public string $name = '';

    #[Validate('required|string|min:3|max:4')]
    public string $currency_code = 'USD';

    #[Validate('required|numeric|min:0.0001')]
    public string $initial_listing_amount = '0.0001';

    #[Validate('required|numeric|min:0.0001')]
    public string $current_amount = '0.0001';

    #[Validate('required|integer|min:1')]
    public int $min_shares = 1;

    #[Validate('required|integer|min:0')]
    public int $max_shares_purchase = 0;

    #[Validate('required|integer|min:0')]
    public int $max_shares_holding = 0;

    #[Validate('nullable|image|mimes:jpg,jpeg,png,webp|max:2048')]
    public $logo;

    public ?string $currentLogo = null;

    #[Validate('nullable|string|max:3000')]
    public string $description = '';

    #[Validate('required|numeric|min:0.0001')]
    public string $log_price = '';

    #[Validate('nullable|numeric')]
    public string $log_change_percent = '';

    #[Validate('required|string|in:manual,system,api')]
    public string $log_source = 'manual';

    public function mount(int $id): void
    {
        $this->security = Security::query()->findOrFail($id);

        $this->trading_name = $this->security->trading_name;
        $this->name = (string) $this->security->name;
        $this->currency_code = (string) $this->security->currency_code;
        $this->initial_listing_amount = (string) $this->security->initial_listing_amount;
        $this->current_amount = (string) $this->security->current_amount;
        $this->min_shares = (int) $this->security->min_shares;
        $this->max_shares_purchase = (int) $this->security->max_shares_purchase;
        $this->max_shares_holding = (int) $this->security->max_shares_holding;
        $this->currentLogo = $this->security->logo;
        $this->description = $this->security->description ?? '';
        $this->log_price = number_format((float) $this->security->latestPrice(), 4, '.', '');
    }

    #[Computed]
    public function logs()
    {
        if (! $this->security) {
            return collect();
        }

        return SecurityLog::query()
            ->where('security_id', $this->security->id)
            ->latest('logged_at')
            ->limit(25)
            ->get();
    }

    public function removeCurrentLogo(): void
    {
        if ($this->currentLogo) {
            MediaForge::delete($this->currentLogo);
        }

        $this->currentLogo = null;

        if ($this->security) {
            $this->security->update(['logo' => null]);
            $this->security->refresh();
        }

        $this->notifySuccess(__('Security logo removed.'));
    }

    public function save(): void
    {
        $this->validate([
            'trading_name' => 'required|string|max:20|unique:securities,trading_name,'.$this->security->id,
            'name' => 'nullable|string|max:255',
            'currency_code' => 'required|string|min:3|max:4',
            'initial_listing_amount' => 'required|numeric|min:0.0001',
            'current_amount' => 'required|numeric|min:0.0001',
            'min_shares' => 'required|integer|min:1',
            'max_shares_purchase' => 'required|integer|min:0',
            'max_shares_holding' => 'required|integer|min:0',
            'logo' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'description' => 'nullable|string|max:3000',
        ]);

        if (! $this->security) {
            return;
        }

        $logoPath = $this->currentLogo;

        if ($this->logo) {
            if ($this->currentLogo) {
                MediaForge::delete($this->currentLogo);
            }

            $logoPath = MediaForge::upload($this->logo)
                ->useYearFolder(true)
                ->randomizeFileName(true)
                ->to('securities')
                ->resize(300, 300)
                ->run();
        }

        $this->security->update([
            'name' => $this->name !== '' ? $this->name : strtoupper($this->trading_name),
            'trading_name' => strtoupper($this->trading_name),
            'logo' => $logoPath,
            'currency_code' => strtoupper($this->currency_code),
            'initial_listing_amount' => (float) $this->initial_listing_amount,
            'current_amount' => (float) $this->current_amount,
            'min_shares' => $this->min_shares,
            'max_shares_purchase' => $this->max_shares_purchase,
            'max_shares_holding' => $this->max_shares_holding,
            'description' => $this->description ?: null,
        ]);

        $this->security->refresh();
        $this->currentLogo = $this->security->logo;

        $this->notifySuccess(__('Security updated successfully.'));
    }

    public function addPriceLog(): void
    {
        $this->validate([
            'log_price' => 'required|numeric|min:0.0001',
            'log_change_percent' => 'nullable|numeric',
            'log_source' => 'required|string|in:manual,system,api',
        ]);

        if (! $this->security) {
            return;
        }

        $price = (float) $this->log_price;
        $latestPrice = $this->security->latestPrice();
        $changePercent = Security::calculateChangePercent($latestPrice, $price);

        if ($this->log_change_percent !== '') {
            $changePercent = (float) $this->log_change_percent;
        }

        $this->security->updateQuietly([
            'current_amount' => $price,
        ]);

        $this->security->recordPriceLog(
            $price,
            $changePercent,
            $this->log_source
        );

        $this->security->refresh();
        $this->current_amount = (string) $this->security->current_amount;
        $this->log_price = number_format($price, 4, '.', '');
        $this->log_change_percent = '';
        unset($this->logs);

        $this->notifySuccess(__('Security price log added.'));
    }
}; ?>

<div>
    <x-admin-panel>
        <x-slot name="header">{{ __('Edit Security') }}</x-slot>
        <x-slot name="desc">{{ __('Update security details and manually add price history logs.') }}</x-slot>
        <x-slot name="button">
            <a href="{{ route('admin.securities.index') }}"
                class="bg-black dark:bg-gray-700 text-white hover:bg-gray-800 dark:hover:bg-gray-600 px-3 py-2 rounded-md float-right text-sm font-bold">
                {{ __('Go Back') }}
            </a>
        </x-slot>

        <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg px-4 py-5 mb-5 sm:p-6">
            {!! $this->renderNotification() !!}

            <form wire:submit="save" class="space-y-6">
                <div class="grid grid-cols-1 gap-4 md:grid-cols-2">
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Ticker Name') }}</label>
                        <input type="text" wire:model="trading_name"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('trading_name') }}</span>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Display Name') }}</label>
                        <input type="text" wire:model="name"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('name') }}</span>
                    </div>

                    <div class="md:col-span-2">
                        <label class="block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Logo') }}</label>
                        <input type="file" wire:model="logo" accept=".jpg,.jpeg,.png,.webp"
                            class="mt-1 block w-full cursor-pointer px-3 py-2 text-sm file:mr-4 file:rounded-md file:border-0 file:bg-gray-200 dark:file:bg-gray-600 file:px-3 file:py-1.5 file:text-sm file:font-semibold file:text-gray-900 dark:file:text-gray-100 hover:file:bg-gray-300 dark:hover:file:bg-gray-500 focus:outline-none focus:ring-2 focus:ring-gray-500 focus:ring-offset-2" />
                        <span class="text-red-500 text-sm">{{ $errors->first('logo') }}</span>

                        @if ($logo)
                            <img src="{{ $logo->temporaryUrl() }}" class="mt-2 h-16 w-16 rounded object-cover" alt="preview" />
                        @elseif ($currentLogo)
                            <div class="mt-2 flex items-center gap-4">
                                <img src="{{ asset($currentLogo) }}" class="h-16 w-16 rounded object-cover" alt="current logo" />
                                <button type="button" wire:click="removeCurrentLogo" class="rounded-md bg-red-600 px-3 py-1 text-xs font-semibold text-white hover:bg-red-500">{{ __('Remove') }}</button>
                            </div>
                        @endif
                    </div>
                </div>

                <div class="grid grid-cols-1 gap-4 md:grid-cols-2">
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Currency Code') }}</label>
                        <input type="text" wire:model="currency_code"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base uppercase text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('currency_code') }}</span>
                    </div>
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Current Price') }}</label>
                        <input type="number" step="0.0001" min="0.0001" wire:model="current_amount"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('current_amount') }}</span>
                    </div>
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Initial Listing Price') }}</label>
                        <input type="number" step="0.0001" min="0.0001" wire:model="initial_listing_amount"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('initial_listing_amount') }}</span>
                    </div>
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Min Shares') }}</label>
                        <input type="number" step="1" min="1" wire:model="min_shares"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('min_shares') }}</span>
                    </div>
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Max Shares Per Purchase (0 = unlimited)') }}</label>
                        <input type="number" step="1" min="0" wire:model="max_shares_purchase"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('max_shares_purchase') }}</span>
                    </div>
                    <div>
                        <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Max Shares Holding (0 = unlimited)') }}</label>
                        <input type="number" step="1" min="0" wire:model="max_shares_holding"
                            class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                        <span class="text-red-500 text-sm">{{ $errors->first('max_shares_holding') }}</span>
                    </div>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Description') }}</label>
                    <textarea wire:model="description" rows="4"
                        class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6"></textarea>
                    <span class="text-red-500 text-sm">{{ $errors->first('description') }}</span>
                </div>

                <div class="flex justify-end">
                    <button type="submit" class="rounded-md bg-indigo-600 px-4 py-2 text-sm font-semibold text-white hover:bg-indigo-500">
                        {{ __('Save Changes') }}
                    </button>
                </div>
            </form>
        </div>

        <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg px-4 py-5 mb-5 sm:p-6">
            <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100">{{ __('Price History Log') }}</h3>
            <p class="mt-1 text-sm text-gray-600 dark:text-gray-300">{{ __('Append manual logs and keep latest price in sync across the system.') }}</p>

            <form wire:submit="addPriceLog" class="mt-4 grid grid-cols-1 gap-4 md:grid-cols-4">
                <div>
                    <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Price') }}</label>
                    <input type="number" step="0.0001" min="0.0001" wire:model="log_price"
                        class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                    <span class="text-red-500 text-sm">{{ $errors->first('log_price') }}</span>
                </div>

                <div>
                    <label class="block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Change % (optional)') }}</label>
                    <input type="number" step="0.0001" wire:model="log_change_percent"
                        class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6" />
                    <span class="text-red-500 text-sm">{{ $errors->first('log_change_percent') }}</span>
                </div>

                <div>
                    <label class="required block text-sm font-medium text-gray-900 dark:text-gray-100">{{ __('Source') }}</label>
                    <select wire:model="log_source"
                        class="mt-1 block w-full rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6">
                        <option value="manual">{{ __('manual') }}</option>
                        <option value="system">{{ __('system') }}</option>
                        <option value="api">{{ __('api') }}</option>
                    </select>
                    <span class="text-red-500 text-sm">{{ $errors->first('log_source') }}</span>
                </div>

                <div class="flex items-end">
                    <button type="submit" class="w-full rounded-md bg-emerald-600 px-4 py-2 text-sm font-semibold text-white hover:bg-emerald-500">
                        {{ __('Add Log') }}
                    </button>
                </div>
            </form>

            <div class="mt-6 overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg">
                <table class="min-w-full divide-y divide-gray-300 dark:divide-gray-600">
                    <thead class="bg-gray-50 dark:bg-gray-700">
                        <tr>
                            <th class="py-3.5 pr-3 pl-4 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Logged At') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Before') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Now') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Change %') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Source') }}</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white dark:bg-gray-800">
                        @php($logs = $this->logs->values())
                        @forelse ($logs as $index => $log)
                            @php($beforeLog = $logs->get($index + 1))
                            <tr class="even:bg-gray-50 dark:even:bg-gray-800/50">
                                <td class="py-4 pr-3 pl-4 text-sm text-gray-900 dark:text-gray-100">{{ optional($log->logged_at)->format('Y-m-d H:i') ?: '-' }}</td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">
                                    {{ $beforeLog ? number_format((float) $beforeLog->price, 4) : '-' }}
                                </td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ number_format((float) $log->price, 4) }}</td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ number_format((float) $log->change_percent, 4) }}</td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $log->source }}</td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="5" class="py-6 text-center text-sm text-gray-500 dark:text-gray-400">{{ __('No logs yet.') }}</td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>
    </x-admin-panel>
</div>
