<?php

use App\Models\InvestmentPackage;
use App\Models\WalletTransaction;
use App\Jobs\Vrm\SendMail;
use App\Mail\AdminDepositMail;
use App\Mail\AdminTransactionMail;
use App\Mail\DepositRequestRejectedMail;
use App\Mail\WalletTransactionMail;
use App\Mail\WithdrawalRequestStatusMail;
use App\Services\Investment\SubscriptionService;
use App\Services\Sms\SmsNotifier;
use App\Services\Vrm\InvestmentEmailService;
use App\Traits\Vrm\Livewire\WithNotifications;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Livewire\WithPagination;

new class extends Component {
    use WithNotifications;
    use WithPagination;

    public string $status = 'pending';

    #[Computed]
    public function depositRequests()
    {
        return WalletTransaction::query()
            ->with(['wallet', 'user'])
            ->whereIn('type', ['deposit_request', 'mpesa_deposit_request'])
            ->when($this->status !== 'all', fn ($query) => $query->where('status', $this->status))
            ->latest()
            ->paginate(12, ['*'], 'deposit_page');
    }

    #[Computed]
    public function withdrawalRequests()
    {
        return WalletTransaction::query()
            ->with(['wallet', 'user'])
            ->where('type', 'withdrawal_request')
            ->when($this->status !== 'all', fn ($query) => $query->where('status', $this->status))
            ->latest()
            ->paginate(12, ['*'], 'withdrawal_page');
    }

    public function approve(int $transactionId): void
    {
        $transaction = WalletTransaction::query()->with(['wallet', 'user'])->find($transactionId);

        if (! $transaction || $transaction->status !== 'pending') {
            $this->notifyError(__('Request not available for approval.'));

            return;
        }

        DB::transaction(function () use ($transaction): void {
            $wallet = $transaction->wallet;

            if (! $wallet) {
                throw new \RuntimeException('Wallet not found.');
            }

            if (in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true)) {
                $wallet->balance = (float) $wallet->balance + (float) $transaction->amount;
            }

            if ($transaction->type === 'withdrawal_request') {
                if (Schema::hasColumn('wallets', 'withdrawal_hold_amount')) {
                    $wallet->withdrawal_hold_amount = max(0, (float) $wallet->withdrawal_hold_amount - (float) $transaction->amount);
                } else {
                    $wallet->used_amount = max(0, (float) $wallet->used_amount - (float) $transaction->amount);
                }
                $wallet->balance = max(0, (float) $wallet->balance - (float) $transaction->amount);
            }

            $wallet->save();

            $transaction->status = 'approved';
            $transaction->approved_by = auth()->id();
            $transaction->approved_at = now();
            $transaction->save();
        });

        $notificationType = in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true) ? 'deposit' : 'withdrawal';
        if ($transaction->user?->email) {
            if ($notificationType === 'withdrawal') {
                SendMail::dispatch($transaction->user->email, mailable: new WithdrawalRequestStatusMail(
                    $transaction->user->name,
                    $transaction->currency_code,
                    (float) $transaction->amount,
                    'approved',
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                    $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user->phone),
                    $transaction->updated_at?->format('M j, Y g:i A'),
                ), notificationType: 'user_transaction');
            } else {
                SendMail::dispatch($transaction->user->email, mailable: new WalletTransactionMail(
                    $transaction->user->name,
                    $notificationType,
                    $transaction->currency_code,
                    (float) $transaction->amount,
                    'approved',
                    $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user->phone),
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                    $transaction->updated_at?->format('M j, Y g:i A'),
                ), notificationType: 'user_transaction');
            }
        }
        app(SmsNotifier::class)->sendToPhone(
            $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user?->phone),
            ucfirst($notificationType).' request approved.',
            'user_transaction'
        );

        $adminEmail = config('mail.admin_address');
        if (in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true) && filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminDepositMail(
                    $transaction->user?->name ?? 'Unknown',
                    $transaction->user?->email ?? 'unknown@example.com',
                    (float) $transaction->amount,
                    $transaction->currency_code,
                    'approved',
                    $transaction->type === 'mpesa_deposit_request' ? 'mpesa' : 'wallet',
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference')
                ), notificationType: 'admin_deposit');

            app(SmsNotifier::class)->sendToAdmins(
                "Deposit request approved for ".($transaction->user?->name ?? 'member').": {$transaction->currency_code} ".number_format((float) $transaction->amount, 2).'.',
                'admin_deposit'
            );
        }

        if ($transaction->type === 'withdrawal_request' && filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminTransactionMail(
                $transaction->user?->name ?? 'Unknown',
                $transaction->user?->email ?? 'unknown@example.com',
                (float) $transaction->amount,
                $transaction->currency_code,
                'withdrawal',
                'approved',
                (string) data_get($transaction->meta, 'channel', 'internal'),
                $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
            ), notificationType: 'admin_transaction');
        }

        if (in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true)) {
            $meta = $transaction->meta ?? [];
            if (($meta['intent'] ?? '') === SubscriptionService::META_INTENT_SUBSCRIPTION && ! empty($meta['package_id']) && isset($meta['amount'])) {
                $package = InvestmentPackage::query()->find($meta['package_id']);
                $user = $transaction->user;
                $wallet = $transaction->wallet;
                if ($package && $user && $wallet) {
                    $wallet->refresh();
                    try {
                        app(SubscriptionService::class)->createSubscription($user, $wallet, $package, (float) $meta['amount']);
                        $transaction->update(['meta' => array_merge($meta, ['subscription_fulfilled' => true])]);
                        app(InvestmentEmailService::class)->sendSubscriptionCreated(
                            $user->email,
                            $user->name,
                            $package->name,
                            app(\App\Services\DefaultCurrencyService::class)->code(),
                            (float) $meta['amount'],
                        );
                        app(SmsNotifier::class)->sendToPhone(
                            $user->phone,
                            "Subscription created for {$package->name}.",
                            'user_transaction'
                        );
                    } catch (\Throwable $e) {
                        Log::warning('Subscription after deposit approval failed', [
                            'transaction_id' => $transaction->id,
                            'message' => $e->getMessage(),
                        ]);
                    }
                }
            }
        }

        $this->notifySuccess(__('Wallet request approved.'));
    }

    public function reject(int $transactionId): void
    {
        $transaction = WalletTransaction::query()->with(['wallet', 'user'])->find($transactionId);

        if (! $transaction || $transaction->status !== 'pending') {
            $this->notifyError(__('Request not available for rejection.'));

            return;
        }

        DB::transaction(function () use ($transaction): void {
            $wallet = $transaction->wallet;

            if (! $wallet) {
                throw new \RuntimeException('Wallet not found.');
            }

            if ($transaction->type === 'withdrawal_request') {
                if (Schema::hasColumn('wallets', 'withdrawal_hold_amount')) {
                    $wallet->withdrawal_hold_amount = max(0, (float) $wallet->withdrawal_hold_amount - (float) $transaction->amount);
                } else {
                    $wallet->used_amount = max(0, (float) $wallet->used_amount - (float) $transaction->amount);
                }
                $wallet->save();
            }

            $transaction->status = 'rejected';
            $transaction->approved_by = auth()->id();
            $transaction->approved_at = now();
            $transaction->save();
        });

        $notificationType = in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true) ? 'deposit' : 'withdrawal';
        if ($transaction->user?->email) {
            if ($notificationType === 'deposit') {
                SendMail::dispatch($transaction->user->email, mailable: new DepositRequestRejectedMail(
                    $transaction->user->name,
                    $transaction->currency_code,
                    (float) $transaction->amount,
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                ), notificationType: 'user_transaction');
            } elseif ($notificationType === 'withdrawal') {
                SendMail::dispatch($transaction->user->email, mailable: new WithdrawalRequestStatusMail(
                    $transaction->user->name,
                    $transaction->currency_code,
                    (float) $transaction->amount,
                    'rejected',
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                    $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user->phone),
                    $transaction->updated_at?->format('M j, Y g:i A'),
                ), notificationType: 'user_transaction');
            } else {
                SendMail::dispatch($transaction->user->email, mailable: new WalletTransactionMail(
                    $transaction->user->name,
                    $notificationType,
                    $transaction->currency_code,
                    (float) $transaction->amount,
                    'rejected',
                    $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user->phone),
                    $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
                    $transaction->updated_at?->format('M j, Y g:i A'),
                ), notificationType: 'user_transaction');
            }
        }
        app(SmsNotifier::class)->sendToPhone(
            $transaction->phone_number ?: data_get($transaction->meta, 'phone_number', $transaction->user?->phone),
            ucfirst($notificationType).' request rejected.',
            'user_transaction'
        );

        $adminEmail = config('mail.admin_address');
        if (in_array($transaction->type, ['deposit_request', 'mpesa_deposit_request'], true) && filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminDepositMail(
                $transaction->user?->name ?? 'Unknown',
                $transaction->user?->email ?? 'unknown@example.com',
                (float) $transaction->amount,
                $transaction->currency_code,
                'rejected',
                $transaction->type === 'mpesa_deposit_request' ? 'mpesa' : 'wallet',
                $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
            ), notificationType: 'admin_deposit');

            app(SmsNotifier::class)->sendToAdmins(
                "Deposit request rejected for ".($transaction->user?->name ?? 'member').": {$transaction->currency_code} ".number_format((float) $transaction->amount, 2).'.',
                'admin_deposit'
            );
        }

        if ($transaction->type === 'withdrawal_request' && filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminTransactionMail(
                $transaction->user?->name ?? 'Unknown',
                $transaction->user?->email ?? 'unknown@example.com',
                (float) $transaction->amount,
                $transaction->currency_code,
                'withdrawal',
                'rejected',
                (string) data_get($transaction->meta, 'channel', 'internal'),
                $transaction->transaction_code ?? data_get($transaction->meta, 'reference'),
            ), notificationType: 'admin_transaction');
        }

        $this->notifySuccess(__('Wallet request rejected.'));
    }
}; ?>

<div>
    <x-admin-panel>
        <x-slot name="header">{{ __('Wallet Requests') }}</x-slot>
        <x-slot name="desc">{{ __('Approve or reject member deposit and withdrawal requests.') }}</x-slot>

        {!! $this->renderNotification() !!}

        <div class="mb-4">
            <label class="block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Status Filter') }}</label>
            <select wire:model.live="status"
                class="mt-1 block w-full max-w-xs rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6">
                <option value="pending">{{ __('Pending') }}</option>
                <option value="approved">{{ __('Approved') }}</option>
                <option value="rejected">{{ __('Rejected') }}</option>
                <option value="all">{{ __('All') }}</option>
            </select>
        </div>

        {{-- Deposit requests --}}
        <div class="mb-10">
            <h2 class="text-base font-semibold text-gray-900 dark:text-gray-100 mb-3">{{ __('Deposit Requests') }}</h2>
            <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg">
                <table class="min-w-full divide-y divide-gray-300 dark:divide-gray-600">
                    <thead class="bg-gray-50 dark:bg-gray-700">
                        <tr>
                            <th class="py-3.5 pr-3 pl-4 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Name') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Phone Number') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Amount') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Transaction Code') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Date & Time') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Status') }}</th>
                            <th class="py-3.5 pr-4 pl-3 text-right text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Actions') }}</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white dark:bg-gray-800">
                        @forelse ($this->depositRequests as $request)
                            <tr class="even:bg-gray-50 dark:even:bg-gray-800/50">
                                <td class="py-4 pr-3 pl-4 text-sm text-gray-900 dark:text-gray-100">{{ $request->user?->name ?? '-' }}</td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $request->phone_number ?? data_get($request->meta, 'phone_number', $request->user?->phone) ?? '-' }}</td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $request->currency_code }} {{ number_format((float) $request->amount, 2) }}</td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $request->transaction_code ?? data_get($request->meta, 'reference') ?? '-' }}</td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $request->created_at?->format('M j, Y g:i A') ?? '-' }}</td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ ucfirst($request->status) }}</td>
                                <td class="py-4 pr-4 pl-3 text-right">
                                    @if ($request->status === 'pending')
                                        <button type="button" wire:click="$js.approveRequest({{ $request->id }})"
                                            class="inline-flex items-center rounded-md bg-green-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-green-500">{{ __('Approve') }}</button>
                                        <button type="button" wire:click="$js.rejectRequest({{ $request->id }})"
                                            class="inline-flex items-center rounded-md bg-red-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-red-500">{{ __('Reject') }}</button>
                                    @else
                                        <span class="text-xs text-gray-500">{{ __('Processed') }}</span>
                                    @endif
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="7" class="py-6 text-center text-sm text-gray-500 dark:text-gray-400">{{ __('No deposit requests found.') }}</td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
            <div class="mt-4">{{ $this->depositRequests->links() }}</div>
        </div>

        {{-- Withdrawal requests --}}
        <div>
            <h2 class="text-base font-semibold text-gray-900 dark:text-gray-100 mb-3">{{ __('Withdrawal Requests') }}</h2>
            <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg">
                <table class="min-w-full divide-y divide-gray-300 dark:divide-gray-600">
                    <thead class="bg-gray-50 dark:bg-gray-700">
                        <tr>
                            <th class="py-3.5 pr-3 pl-4 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Member') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Amount') }}</th>
                            <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Status') }}</th>
                            <th class="py-3.5 pr-4 pl-3 text-right text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Actions') }}</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white dark:bg-gray-800">
                        @forelse ($this->withdrawalRequests as $request)
                            <tr class="even:bg-gray-50 dark:even:bg-gray-800/50">
                                <td class="py-4 pr-3 pl-4 text-sm text-gray-900 dark:text-gray-100">{{ $request->user?->name ?? '-' }}</td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $request->currency_code }} {{ number_format((float) $request->amount, 2) }}</td>
                                <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ ucfirst($request->status) }}</td>
                                <td class="py-4 pr-4 pl-3 text-right">
                                    @if ($request->status === 'pending')
                                        <button type="button" wire:click="$js.approveRequest({{ $request->id }})"
                                            class="inline-flex items-center rounded-md bg-green-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-green-500">{{ __('Approve') }}</button>
                                        <button type="button" wire:click="$js.rejectRequest({{ $request->id }})"
                                            class="inline-flex items-center rounded-md bg-red-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-red-500">{{ __('Reject') }}</button>
                                    @else
                                        <span class="text-xs text-gray-500">{{ __('Processed') }}</span>
                                    @endif
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="4" class="py-6 text-center text-sm text-gray-500 dark:text-gray-400">{{ __('No withdrawal requests found.') }}</td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
            <div class="mt-4">{{ $this->withdrawalRequests->links() }}</div>
        </div>
    </x-admin-panel>

    @script
        <script>
            $js('approveRequest', async (requestId) => {
                const result = await Swal.fire({
                    title: 'Are you sure you want to approve this request?',
                    icon: 'question',
                    showCancelButton: true,
                });

                if (result.isConfirmed) {
                    $wire.approve(requestId);
                }
            });

            $js('rejectRequest', async (requestId) => {
                const result = await Swal.fire({
                    title: 'Are you sure you want to reject this request?',
                    icon: 'warning',
                    showCancelButton: true,
                });

                if (result.isConfirmed) {
                    $wire.reject(requestId);
                }
            });
        </script>
    @endscript
</div>
