<?php

use App\Models\SecurityWatchlist;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Livewire\WithPagination;

new class extends Component {
    use WithNotifications;
    use WithPagination;

    public string $search = '';

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    #[Computed]
    public function watchlists()
    {
        return SecurityWatchlist::query()
            ->with(['user', 'security'])
            ->when($this->search !== '', function ($query): void {
                $query->where(function ($innerQuery): void {
                    $innerQuery
                        ->whereHas('user', function ($userQuery): void {
                            $userQuery->where('name', 'like', "%{$this->search}%")
                                ->orWhere('email', 'like', "%{$this->search}%");
                        })
                        ->orWhereHas('security', function ($securityQuery): void {
                            $securityQuery->where('name', 'like', "%{$this->search}%")
                                ->orWhere('trading_name', 'like', "%{$this->search}%");
                        });
                });
            })
            ->latest()
            ->paginate(12);
    }

    public function remove(int $watchlistId): void
    {
        $watchlist = SecurityWatchlist::query()->find($watchlistId);

        if (! $watchlist) {
            $this->notifyError(__('Watchlist item not found.'));

            return;
        }

        $watchlist->delete();

        $this->notifySuccess(__('Watchlist item removed.'));
    }
}; ?>

<div>
    <x-admin-panel>
        <x-slot name="header">{{ __('Watchlists') }}</x-slot>
        <x-slot name="desc">{{ __('Review and manage member security watchlists.') }}</x-slot>

        <div class="mb-4">
            <label class="block text-sm/6 font-medium text-gray-900 dark:text-gray-100">{{ __('Search') }}</label>
            <input type="text" wire:model.live.debounce.300ms="search"
                class="mt-1 block w-full max-w-md rounded-md bg-white dark:bg-gray-700 px-3 py-1.5 text-base text-gray-900 dark:text-gray-100 outline-1 -outline-offset-1 outline-gray-300 dark:outline-gray-600 placeholder:text-gray-400 dark:placeholder:text-gray-500 focus:outline-2 focus:-outline-offset-2 focus:outline-indigo-600 sm:text-sm/6"
                placeholder="{{ __('Search by member or security...') }}" />
        </div>

        {!! $this->renderNotification() !!}

        <div class="overflow-hidden shadow-sm ring-1 ring-black/5 dark:ring-white/10 sm:rounded-lg mt-2">
            <table class="min-w-full divide-y divide-gray-300 dark:divide-gray-600">
                <thead class="bg-gray-50 dark:bg-gray-700">
                    <tr>
                        <th class="py-3.5 pr-3 pl-4 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Member') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Security') }}</th>
                        <th class="py-3.5 px-3 text-left text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Added On') }}</th>
                        <th class="py-3.5 pr-4 pl-3 text-right text-sm font-semibold text-gray-900 dark:text-gray-100">{{ __('Actions') }}</th>
                    </tr>
                </thead>
                <tbody class="bg-white dark:bg-gray-800">
                    @forelse ($this->watchlists as $item)
                        <tr class="even:bg-gray-50 dark:even:bg-gray-800/50">
                            <td class="py-4 pr-3 pl-4 text-sm text-gray-900 dark:text-gray-100">
                                <p class="font-semibold">{{ $item->user?->name ?? '-' }}</p>
                                <p class="text-xs text-gray-500">{{ $item->user?->email ?? '-' }}</p>
                            </td>
                            <td class="py-4 px-3 text-sm text-gray-900 dark:text-gray-100">{{ $item->security?->trading_name ?? '-' }}</td>
                            <td class="py-4 px-3 text-sm text-gray-500 dark:text-gray-400">{{ $item->created_at?->format('M j, Y g:i A') ?? '-' }}</td>
                            <td class="py-4 pr-4 pl-3 text-right">
                                <button type="button" wire:click="$js.confirmRemove({{ $item->id }})"
                                    class="inline-flex items-center rounded-md bg-red-600 px-2.5 py-1 text-xs font-semibold text-white hover:bg-red-500">{{ __('Remove') }}</button>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="4" class="py-6 text-center text-sm text-gray-500 dark:text-gray-400">{{ __('No watchlist items found.') }}</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <div class="mt-4">{{ $this->watchlists->links() }}</div>
    </x-admin-panel>

    @script
        <script>
            $js('confirmRemove', async (watchlistId) => {
                const result = await Swal.fire({
                    title: 'Remove watchlist item?',
                    icon: 'warning',
                    showCancelButton: true,
                });

                if (result.isConfirmed) {
                    $wire.remove(watchlistId);
                }
            });
        </script>
    @endscript
</div>
