<?php

use App\Models\PortfolioHolding;
use App\Models\Security;
use App\Models\SecurityOrder;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Jobs\Vrm\SendMail;
use App\Mail\AdminTransactionMail;
use App\Services\DefaultCurrencyService;
use App\Services\Sms\SmsNotifier;
use App\Services\Trading\TradingSettingsService;
use App\Traits\Vrm\Livewire\WithNotifications;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Component;

new #[Layout('layouts.trading')] class extends Component {
    use WithNotifications;

    public int $securityId;
    public string $amount = '';
    public string $investmentStep = 'form';
    public float $submittedInvestmentAmount = 0.0;
    public int $submittedInvestmentQuantity = 0;

    public function mount(int $security): void
    {
        $this->useToasts();
        $this->securityId = $security;

        $exists = Security::query()
            ->where('is_active', true)
            ->whereKey($this->securityId)
            ->exists();

        abort_unless($exists, 404);
    }

    #[Computed]
    public function security(): Security
    {
        return Security::query()
            ->where('is_active', true)
            ->findOrFail($this->securityId);
    }

    #[Computed]
    public function currentSharePrice(): float
    {
        return max(0.0001, (float) $this->security->latestPrice());
    }

    #[Computed]
    public function wallet(): Wallet
    {
        return Wallet::query()->firstOrCreate(
            ['user_id' => auth()->id()],
            ['currency_code' => app(DefaultCurrencyService::class)->code()]
        );
    }

    /** System default currency from ATU (single currency across the app). */
    #[Computed]
    public function currencyCode(): string
    {
        return app(DefaultCurrencyService::class)->code();
    }

    #[Computed]
    public function isMarketOpen(): bool
    {
        return app(TradingSettingsService::class)->isMarketOpen();
    }

    #[Computed]
    public function holding(): array
    {
        $holding = PortfolioHolding::query()
            ->where('user_id', auth()->id())
            ->where('security_id', $this->securityId)
            ->first();

        if (! $holding) {
            return [
                'quantity' => 0,
                'current_value' => 0,
                'total_invested' => 0,
            ];
        }

        $quantity = (int) $holding->quantity;
        $totalInvested = (float) $holding->total_invested;

        return [
            'quantity' => $quantity,
            'current_value' => $this->currentSharePrice * $quantity,
            'total_invested' => $totalInvested,
        ];
    }

    #[Computed]
    public function hasInvested(): bool
    {
        return $this->holding['quantity'] > 0 || $this->holding['total_invested'] > 0;
    }

    #[Computed]
    public function investmentLogs(): array
    {
        return SecurityOrder::query()
            ->where('user_id', auth()->id())
            ->where('security_id', $this->securityId)
            ->where('order_type', 'buy')
            ->latest('traded_at')
            ->limit(10)
            ->get()
            ->map(function (SecurityOrder $order): array {
                return [
                    'id' => $order->id,
                    'amount' => (float) $order->total_amount,
                    'quantity' => (int) $order->quantity,
                    'status' => (string) $order->status,
                    'date' => $order->traded_at?->format('M d, Y h:i A') ?? 'N/A',
                ];
            })
            ->all();
    }

    public function goToInvestmentConfirmation(): void
    {
        $preparedOrder = $this->prepareInvestmentOrder();

        if ($preparedOrder === null) {
            return;
        }

        $this->investmentStep = 'confirm';
    }

    public function backToInvestmentForm(): void
    {
        $this->investmentStep = 'form';
    }

    public function submitInvestment(): bool
    {
        if ($this->investmentStep !== 'confirm') {
            $this->goToInvestmentConfirmation();

            return false;
        }

        $preparedOrder = $this->prepareInvestmentOrder();

        if ($preparedOrder === null) {
            $this->investmentStep = 'form';

            return false;
        }

        /** @var \App\Models\Security $security */
        $security = $preparedOrder['security'];
        /** @var \App\Models\Wallet $wallet */
        $wallet = $preparedOrder['wallet'];
        $quantity = $preparedOrder['quantity'];
        $totalAmount = $preparedOrder['total_amount'];
        $pricePerShare = $preparedOrder['price_per_share'];

        DB::transaction(function () use ($security, $wallet, $quantity, $totalAmount, $pricePerShare): void {
            SecurityOrder::query()->create([
                'user_id' => auth()->id(),
                'security_id' => $security->id,
                'wallet_id' => $wallet->id,
                'order_type' => 'buy',
                'status' => 'pending',
                'price_per_share' => $pricePerShare,
                'quantity' => $quantity,
                'volume' => $quantity,
                'total_amount' => $totalAmount,
                'currency_code' => app(DefaultCurrencyService::class)->code(),
                'traded_at' => now(),
                'profit_loss_amount' => 0,
                'profit_loss_percent' => 0,
                'is_active' => true,
            ]);

            $wallet->used_amount = (float) $wallet->used_amount + $totalAmount;
            $wallet->save();

            WalletTransaction::query()->create([
                'wallet_id' => $wallet->id,
                'user_id' => auth()->id(),
                'type' => 'order_debit',
                'status' => 'pending',
                'amount' => $totalAmount,
                'currency_code' => app(DefaultCurrencyService::class)->code(),
                'description' => "Pending buy order for {$security->trading_name}",
                'meta' => ['security_id' => $security->id, 'quantity' => $quantity, 'order_type' => 'buy'],
            ]);
        });

        $adminEmail = config('mail.admin_address');
        if (filled($adminEmail)) {
            SendMail::dispatch((string) $adminEmail, mailable: new AdminTransactionMail(
                    auth()->user()->name,
                    auth()->user()->email,
                    $totalAmount,
                    app(DefaultCurrencyService::class)->code(),
                    'shares_buy',
                    'submitted',
                    'trade'
                ), notificationType: 'admin_transaction');
        }
        app(SmsNotifier::class)->sendToPhone(
            auth()->user()->phone,
            "Investment request submitted for {$security->trading_name}.",
            'user_transaction'
        );

        $this->submittedInvestmentAmount = $totalAmount;
        $this->submittedInvestmentQuantity = $quantity;
        $this->amount = '';
        $this->investmentStep = 'result';
        $this->notifySuccess(__('Investment request submitted and pending admin approval.'));
        $this->dispatch('security-order-submitted');

        return true;
    }

    public function resetInvestmentFlow(): void
    {
        $this->amount = '';
        $this->investmentStep = 'form';
        $this->submittedInvestmentAmount = 0.0;
        $this->submittedInvestmentQuantity = 0;
    }

    protected function prepareInvestmentOrder(): ?array
    {
        $this->validate([
            'amount' => ['required', 'numeric', 'min:0.01'],
        ]);

        if (! $this->isMarketOpen) {
            $this->notifyError(__('Market is currently closed. Trading is unavailable.'));

            return null;
        }

        $security = $this->security;
        $wallet = $this->wallet;
        $amount = (float) $this->amount;
        $pricePerShare = $this->currentSharePrice;
        $quantity = (int) floor($amount / $pricePerShare);

        if ($quantity <= 0) {
            $this->notifyError(__('Amount is too low for this security price.'));

            return null;
        }

        if ($quantity < $security->min_shares) {
            $this->notifyError(__('Quantity must be at least :min shares.', ['min' => $security->min_shares]));

            return null;
        }

        if ($security->max_shares_purchase > 0 && $quantity > $security->max_shares_purchase) {
            $this->notifyError(__('Quantity exceeds max shares per purchase.'));

            return null;
        }

        $existingHoldingQuantity = (int) auth()->user()
            ->portfolioHoldings()
            ->where('security_id', $security->id)
            ->value('quantity');

        if ($security->max_shares_holding > 0 && ($existingHoldingQuantity + $quantity) > $security->max_shares_holding) {
            $this->notifyError(__('Order exceeds your maximum allowed holding for this security.'));

            return null;
        }

        $totalAmount = $pricePerShare * $quantity;

        if ($wallet->available_balance < $totalAmount) {
            $this->notifyError(__('You do not have enough available balance. Please make a deposit first.'));

            return null;
        }

        return [
            'security' => $security,
            'wallet' => $wallet,
            'quantity' => $quantity,
            'total_amount' => $totalAmount,
            'price_per_share' => $pricePerShare,
        ];
    }
}; ?>

<div>
    @include('partials.trading.topbar', ['title' => 'INVEST'])

    <div class="px-4 py-4 space-y-4">
        @if (!empty($this->notification))
            <div x-data="{ show: true }" x-show="show" x-transition:enter="transform transition ease-out duration-300"
                x-transition:enter-start="translate-y-3 opacity-0" x-transition:enter-end="translate-y-0 opacity-100"
                x-transition:leave="transform transition ease-in duration-200"
                x-transition:leave-start="translate-y-0 opacity-100" x-transition:leave-end="translate-y-3 opacity-0"
                x-init="setTimeout(() => {
    show = false;
    $wire.clearNotification();
}, 3500)"
                class="fixed inset-x-0 bottom-24 z-[96] px-4 lg:bottom-6 [&_.toast]:mx-auto [&_.toast]:max-w-sm [&_.toast]:w-full">
                {!! $this->renderNotification() !!}
            </div>
        @endif

        <a href="{{ route('member.trade-now') }}" wire:navigate class="inline-flex items-center gap-2 text-sm font-medium text-primary">
            <i class="fa-solid fa-arrow-left h-4 w-4"></i>
            {{ __('Back to Trade List') }}
        </a>

        @if (! $this->isMarketOpen)
            <div class="rounded-lg border border-destructive/50 bg-destructive/10 px-3 py-2 text-sm text-destructive">
                {{ __('Market is closed for today.') }}
            </div>
        @endif

        <div class="bg-card rounded-xl border border-border p-4 space-y-4">
            <div class="flex items-center justify-between gap-3">
                <div class="flex items-center gap-3 min-w-0">
                    <div class="w-10 h-10 bg-secondary rounded-full flex items-center justify-center shrink-0 overflow-hidden">
                        @if (!empty($this->security->logo))
                            <img src="{{ asset($this->security->logo) }}" alt="" class="size-10 rounded-full object-cover" />
                        @else
                            <span class="text-xs font-bold text-foreground">{{ strtoupper(substr($this->security->trading_name, 0, 2)) }}</span>
                        @endif
                    </div>
                    <div class="min-w-0">
                        <p class="text-foreground font-bold text-sm">{{ $this->security->trading_name }}</p>
                        <p class="text-muted-foreground text-xs truncate">{{ $this->security->name }}</p>
                    </div>
                </div>

                <div class="text-right">
                    <p class="text-foreground font-bold text-sm">{{ $this->currencyCode }} {{ number_format($this->currentSharePrice, 2) }}</p>
                    <p class="text-xs text-muted-foreground">{{ __('Per share') }}</p>
                </div>
            </div>

            <div class="grid grid-cols-2 gap-3 text-sm">
                <div class="rounded-lg border border-border p-3">
                    <p class="text-xs text-muted-foreground">{{ __('Available Balance') }}</p>
                    <p class="font-semibold text-foreground">{{ $this->currencyCode }} {{ number_format($this->wallet->available_balance, 2) }}</p>
                </div>
                <div class="rounded-lg border border-border p-3">
                    <p class="text-xs text-muted-foreground">{{ __('Price Per Share') }}</p>
                    <p class="font-semibold text-foreground">{{ $this->currencyCode }} {{ number_format($this->currentSharePrice, 2) }}</p>
                </div>
                <div class="rounded-lg border border-border p-3">
                    <p class="text-xs text-muted-foreground">{{ __('Current Shares') }}</p>
                    <p class="font-semibold text-foreground">{{ $this->holding['quantity'] }}</p>
                </div>
            </div>

            @if ($investmentStep === 'form')
                @php
                    $amountValue = (float) ($amount !== '' ? $amount : 0);
                    $estimatedShares = $amountValue > 0 ? (int) floor($amountValue / $this->currentSharePrice) : 0;
                @endphp

                <div class="space-y-2">
                    <label for="investment-amount" class="block text-xs text-muted-foreground">{{ __('Amount to invest') }}</label>
                    <input
                        id="investment-amount"
                        type="number"
                        min="0.01"
                        step="0.01"
                        wire:model.live.debounce.300ms="amount"
                        placeholder="0.00"
                        class="w-full rounded-md border border-border bg-background px-3 py-2 text-sm text-foreground"
                    />
                    @error('amount')
                        <p class="text-xs text-destructive">{{ $message }}</p>
                    @enderror
                    <p class="text-xs text-muted-foreground">{{ __('Estimated shares') }}: <span class="font-semibold text-foreground">{{ max(0, $estimatedShares) }}</span></p>
                </div>

                <button
                    type="button"
                    wire:click="goToInvestmentConfirmation"
                    @disabled(! $this->isMarketOpen)
                    class="w-full rounded-xl px-4 py-2.5 text-sm font-semibold {{ $this->isMarketOpen ? 'bg-[#2CB34A] text-white' : 'cursor-not-allowed bg-gray-400 text-white' }}"
                >
                    {{ __('Continue to Confirmation') }}
                </button>
            @elseif ($investmentStep === 'confirm')
                @php
                    $confirmedAmount = (float) $amount;
                    $confirmedShares = $confirmedAmount > 0 ? (int) floor($confirmedAmount / $this->currentSharePrice) : 0;
                @endphp

                <div class="min-h-[68vh] flex flex-col gap-6">
                    <div class="rounded-3xl border border-border bg-card p-5 space-y-6">
                        <div class="mx-auto size-12 rounded-full bg-[#2CB34A] text-white flex items-center justify-center">
                            <i class="fa-solid fa-sack-dollar text-xl"></i>
                        </div>

                        <p class="text-center text-md font-bold tracking-wide text-foreground">{{ __('CONFIRM BUY ORDER') }}</p>

                        <div class="space-y-5 text-sm">
                            <div>
                                <p class="text-[11px] font-semibold uppercase tracking-wide text-muted-foreground">{{ __('Security') }}</p>
                                <p class="text-md font-bold text-foreground">{{ $this->security->trading_name }}</p>
                            </div>
                            <div>
                                <p class="text-[11px] font-semibold uppercase tracking-wide text-muted-foreground">{{ __('Amount') }}</p>
                                <p class="text-md font-bold text-foreground">{{ $this->currencyCode }} {{ number_format($confirmedAmount, 2) }}</p>
                            </div>
                            <div>
                                <p class="text-[11px] font-semibold uppercase tracking-wide text-muted-foreground">{{ __('Estimated Shares') }}</p>
                                <p class="text-md font-bold text-foreground">{{ max(0, $confirmedShares) }}</p>
                            </div>
                            <div class="flex items-end justify-between gap-3">
                                <div>
                                    <p class="text-[11px] font-semibold uppercase tracking-wide text-muted-foreground">{{ __('Available Balance') }}</p>
                                    <p class="text-md font-bold text-foreground">{{ $this->currencyCode }} {{ number_format($this->wallet->available_balance, 2) }}</p>
                                </div>
                                <button type="button" wire:click="backToInvestmentForm"
                                    class="inline-flex size-10 items-center justify-center rounded-full bg-secondary text-muted-foreground">
                                    <i class="fa-solid fa-pen text-md"></i>
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="grid grid-cols-2 gap-2 mt-auto">
                        <button
                            type="button"
                            wire:click="backToInvestmentForm"
                            class="w-full rounded-xl border border-border bg-background px-4 py-2.5 text-sm font-semibold text-foreground"
                        >
                            {{ __('Back') }}
                        </button>
                        <button
                            type="button"
                            wire:click="submitInvestment"
                            wire:loading.attr="disabled"
                            wire:target="submitInvestment"
                            class="w-full rounded-xl bg-[#2CB34A] px-4 py-2.5 text-sm font-semibold text-white disabled:cursor-not-allowed disabled:opacity-80"
                        >
                            <span wire:loading.remove wire:target="submitInvestment">{{ __('Confirm Buy') }}</span>
                            <span wire:loading wire:target="submitInvestment" class="inline-flex items-center gap-2">
                                <i class="fa-solid fa-spinner animate-spin text-sm"></i>
                                {{ __('Submitting') }}
                            </span>
                        </button>
                    </div>
                </div>
            @else
                <div class="min-h-[68vh] flex flex-col">
                    <div class="pt-6 space-y-3 text-center">
                        <div class="mx-auto size-12 rounded-full bg-[#2CB34A] text-white flex items-center justify-center">
                            <i class="fa-solid fa-check text-2xl"></i>
                        </div>
                        <p class="text-md font-extrabold tracking-wide text-foreground">{{ __('BUY REQUEST SUBMITTED') }}</p>
                        <p class="text-md font-extrabold tracking-wide text-foreground">
                            {{ __('AMOUNT: :currency :amount', ['currency' => $this->currencyCode, 'amount' => number_format($submittedInvestmentAmount, 2)]) }}
                        </p>
                    </div>

                    <div class="my-auto text-center space-y-4 px-8">
                        <p class="text-sm text-muted-foreground">{{ __('Security') }}: {{ $this->security->trading_name }}</p>
                        <p class="text-sm text-muted-foreground">{{ __('Shares') }}: {{ $submittedInvestmentQuantity }}</p>
                        <p class="text-md font-medium text-green-600">
                            {{ __('Your investment request has been received and is pending admin approval.') }}
                        </p>
                    </div>

                    <div class="grid grid-cols-2 gap-2 mt-1">
                        <button
                            type="button"
                            wire:click="resetInvestmentFlow"
                            class="w-full rounded-md border border-border bg-background px-4 py-2 text-sm font-semibold text-foreground"
                        >
                            {{ __('Invest Again') }}
                        </button>
                        <a
                            href="{{ route('member.trade-now') }}"
                            wire:navigate
                            class="inline-flex w-full items-center justify-center rounded-md bg-[#2CB34A] px-4 py-2 text-sm font-extrabold uppercase tracking-wide text-white"
                        >
                            {{ __('Done') }}
                        </a>
                    </div>
                </div>
            @endif
        </div>

        @if ($this->hasInvested)
            <div class="bg-card rounded-xl border border-border p-4 space-y-3">
                <div class="flex items-center justify-between">
                    <p class="text-sm font-semibold text-foreground">{{ __('Investment Log') }}</p>
                    <span class="text-xs text-muted-foreground">
                        {{ __('Total Invested') }}: {{ $this->currencyCode }} {{ number_format($this->holding['total_invested'], 2) }}
                    </span>
                </div>

                @forelse ($this->investmentLogs as $log)
                    <div class="rounded-lg border border-border px-3 py-2" wire:key="invest-log-{{ $log['id'] }}">
                        <div class="flex items-center justify-between">
                            <p class="text-sm font-semibold text-foreground">{{ $this->currencyCode }} {{ number_format($log['amount'], 2) }}</p>
                            <span class="text-xs rounded-full px-2 py-0.5 {{ $log['status'] === 'approved' ? 'bg-gain/15 text-gain' : 'bg-secondary text-muted-foreground' }}">
                                {{ ucfirst($log['status']) }}
                            </span>
                        </div>
                        <p class="text-xs text-muted-foreground">{{ __('Shares') }}: {{ $log['quantity'] }}</p>
                        <p class="text-xs text-muted-foreground">{{ __('Invested on') }}: {{ $log['date'] }}</p>
                    </div>
                @empty
                    <p class="text-xs text-muted-foreground">{{ __('No investment logs yet for this security.') }}</p>
                @endforelse
            </div>
        @endif
    </div>

</div>
