<?php

use App\Actions\Fortify\PasswordValidationRules;
use App\Facades\Vrm\MediaForge;
use App\Models\User;
use App\Traits\Vrm\Livewire\WithNotifications;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

return new class extends Component {
    use PasswordValidationRules;
    use WithFileUploads;
    use WithNotifications;

    // Path to upload the photo
    public $uploadedPath = 'admin-users';

    #[Validate('required|string|max:255')]
    public $name = '';

    #[Validate('required|string|email|max:255')]
    public $email = '';

    #[Validate('nullable|string|max:20')]
    public $phone = '';

    #[Validate('nullable|string|max:20')]
    public $whatsapp_number = '';

    #[Validate('required|string')]
    public $password = '';

    #[Validate('required|string')]
    public $password_confirmation = '';

    #[Validate('nullable|image|mimes:jpg,jpeg,png,webp|max:2048')]
    public $photo;

    /**
				 * Create a new admin user.
				 */
    public function create(): void
    {
        // Validate the form using the same rules as CreateNewUser
        $validated = Validator::make(
            [
                'name' => $this->name,
                'email' => $this->email,
                'phone' => $this->phone,
                'whatsapp_number' => $this->whatsapp_number,
                'password' => $this->password,
                'password_confirmation' => $this->password_confirmation,
                'photo' => $this->photo,
            ],
            [
                'name' => ['required', 'string', 'max:255'],
                'email' => ['required', 'string', 'email', 'max:255', Rule::unique(User::class)],
                'phone' => ['nullable', 'string', 'max:20'],
                'whatsapp_number' => ['nullable', 'string', 'max:20'],
                'password' => $this->passwordRules(),
                'photo' => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:2048'],
            ],
        )->validate();

        try {
            // Handle photo upload if provided
            $_photo = null;
            if ($this->photo) {
                try {
                    // Upload photo
                    $_photo = MediaForge::upload($this->photo)->useYearFolder(true)->randomizeFileName(true)->to($this->uploadedPath)->resize(400, 400)->run();
                } catch (\Exception $e) {
                    $this->notifyError(__('Failed to upload photo. Please try again: ' . $e->getMessage()));

                    return;
                }
            }

            // Create the user
            $user = User::create([
                'name' => $validated['name'],
                'email' => $validated['email'],
                'phone' => $validated['phone'] ?? null,
                'password' => $validated['password'],
                'is_active' => true,
            ]);

            // Store whatsapp number in user_meta if provided
            if (!empty($validated['whatsapp_number'])) {
                $user->setMeta('whatsapp_number', $validated['whatsapp_number']);
            }

            // Store photo in user_meta if provided
            if ($_photo) {
                $user->setMeta('photo', $_photo);
            }

            // Attach admin role (role ID 1)
            $user->roles()->attach(1);

            // Reset form
            $this->reset(['name', 'email', 'phone', 'whatsapp_number', 'password', 'password_confirmation', 'photo']);

            // Show success message
            $this->notifySuccess(__('Admin created successfully!'));
        } catch (\Exception $e) {
            $this->notifyError(__('Failed to create admin. Please try again: ' . $e->getMessage()));
        }
    }

    public function cancel(): void
    {
        $this->reset(['name', 'email', 'phone', 'whatsapp_number', 'password', 'password_confirmation', 'photo']);
        $this->notifyInfo(__('Admin creation cancelled!'));
    }

    protected function view($data = [])
    {
        return app('view')->file('/home/ziidfelj/home.ziidi.trade/storage/framework/views/livewire/views/0aa4aa1d.blade.php', $data);
    }
}; 