<?php

use App\Models\SecurityOrder;
use App\Models\User;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Component;

return new class extends Component {
    use WithNotifications;

    public ?User $member = null;
    public string $annualNetRatePercent = '0.0';

    public function mount(int $id): void
    {
        $this->member = User::query()
            ->with(['wallet', 'portfolioHoldings.security'])
            ->findOrFail($id);

        $storedRate = $this->member->getMeta('annual_net_rate_percent', 0);
        $this->annualNetRatePercent = is_numeric($storedRate) ? number_format((float) $storedRate, 1, '.', '') : '0.0';
    }

    public function activate(): void
    {
        if (! $this->member) {
            return;
        }

        $this->member->forceFill([
            'is_active' => true,
            'email_verified_at' => $this->member->email_verified_at ?? now(),
        ])->save();
        $this->member->refresh();

        $this->notifySuccess(__('Member account approved and activated.'));
    }

    public function suspend(): void
    {
        if (! $this->member) {
            return;
        }

        $this->member->update(['is_active' => false]);
        $this->member->refresh();

        $this->notifySuccess(__('Member account suspended.'));
    }

    public function updateAnnualNetRate(): void
    {
        if (! $this->member) {
            return;
        }

        $validated = $this->validate([
            'annualNetRatePercent' => ['required', 'numeric', 'min:0', 'max:1000'],
        ]);

        $rate = round((float) $validated['annualNetRatePercent'], 1);
        $this->member->setMeta('annual_net_rate_percent', $rate);
        $this->annualNetRatePercent = number_format($rate, 1, '.', '');

        $this->notifySuccess(__('Member annual net rate updated.'));
    }

    public function getPendingOrdersCountProperty(): int
    {
        if (! $this->member) {
            return 0;
        }

        return SecurityOrder::query()
            ->where('user_id', $this->member->id)
            ->where('status', 'pending')
            ->count();
    }

    protected function view($data = [])
    {
        return app('view')->file('/home/ziidfelj/home.ziidi.trade/storage/framework/views/livewire/views/2255d21d.blade.php', $data);
    }
}; 