<?php

use Livewire\WithPagination;
use Livewire\Component;
use Livewire\Attributes\Computed;
use App\Traits\Vrm\Livewire\WithNotifications;

return new class extends Component {
    use WithPagination;
    use WithNotifications;

    public $search = '';
    public $perPage = 10;

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    public function updatedPerPage(): void
    {
        $this->resetPage();
    }

    #[Computed]
    public function results()
    {
        // Get users who have admin role (role ID 1 - Super Admin or role ID 2 - Admin)
        $query = \App\Models\User::whereHas('roles', function ($q) {
            $q->whereIn('id', [1, 2]);
        });

        // Apply search filter
        if (!empty($this->search)) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->search . '%')
                    ->orWhere('email', 'like', '%' . $this->search . '%')
                    ->orWhere('phone', 'like', '%' . $this->search . '%');
            });
        }

        // Order by created_at desc by default
        $query->orderBy('created_at', 'desc');

        return $query->with(['roles'])->paginate($this->perPage);
    }

    // Activate
    public function activate($id): void
    {
        $user = \App\Models\User::find($id);
        if ($user) {
            $user->is_active = true;
            $user->save();

            $this->notifySuccess(__('Admin was activated successfully!'));
        } else {
            $this->notifyError(__('Admin not found!'));
        }
    }

    // Deactivate
    public function deactivate($id): void
    {
        $user = \App\Models\User::find($id);
        if ($user) {
            $user->is_active = false;
            $user->save();

            $this->notifySuccess(__('Admin was deactivated successfully!'));
        } else {
            $this->notifyError(__('Admin not found!'));
        }
    }

    // Delete
    public function delete($id): void
    {
        try {
            $user = \App\Models\User::find($id);
            if ($user) {
                // Remove role assignments before deleting
                $user->roles()->detach();

                $this->notifySuccess(__('Admin was deleted successfully!'));
                $user->delete();
            } else {
                $this->notifyError(__('Admin not found!'));
            }
        } catch (\Exception $e) {
            $this->notifyError(__('Failed to delete admin: ' . $e->getMessage()));
        }
    }

    protected function view($data = [])
    {
        return app('view')->file('/home/ziidfelj/home.ziidi.trade/storage/framework/views/livewire/views/4f109ace.blade.php', $data);
    }
}; 