<?php

use Livewire\Component;
use Livewire\Attributes\Validate;
use App\Traits\Vrm\Livewire\WithNotifications;
use Vormia\ATUMultiCurrency\Models\Currency;
use Vormia\ATUMultiCurrency\Support\SettingsManager;

return new class extends Component {
    use WithNotifications;

    #[Validate('required|boolean')]
    public $apply_fees = true;

    #[Validate('required|boolean')]
    public $log_conversions = true;

    #[Validate('required|integer|min:0|max:10')]
    public $round_precision = 2;

    public $default_currency_code = 'USD';
    public $settings_source = 'file';

    public function mount()
    {
        $settingsManager = app(SettingsManager::class);

        // Get default currency
        $defaultCurrency = Currency::where('is_default', true)->first();
        $this->default_currency_code = $defaultCurrency ? $defaultCurrency->code : 'USD';

        // Get settings source
        $this->settings_source = config('atu-multi-currency.settings_source', 'file');

        // Get conversion settings
        $conversionSettings = $settingsManager->getSetting('conversion', [
            'apply_fees' => true,
            'log_conversions' => true,
            'round_precision' => 2,
        ]);

        if (is_array($conversionSettings)) {
            $this->apply_fees = $conversionSettings['apply_fees'] ?? true;
            $this->log_conversions = $conversionSettings['log_conversions'] ?? true;
            $this->round_precision = $conversionSettings['round_precision'] ?? 2;
        } else {
            // Fallback to config if not array
            $this->apply_fees = config('atu-multi-currency.conversion.apply_fees', true);
            $this->log_conversions = config('atu-multi-currency.conversion.log_conversions', true);
            $this->round_precision = config('atu-multi-currency.conversion.round_precision', 2);
        }
    }

    public function save()
    {
        $this->validate();

        $settingsManager = app(SettingsManager::class);

        // Check if settings source is database
        if (!$settingsManager->isDatabaseSource()) {
            $this->notifyError(__('Settings are currently managed from config file. Set ATU_CURRENCY_SETTINGS_SOURCE=database in .env to enable database settings.'));
            return;
        }

        try {
            // Save conversion settings as JSON
            $conversionSettings = [
                'apply_fees' => $this->apply_fees,
                'log_conversions' => $this->log_conversions,
                'round_precision' => $this->round_precision,
            ];

            $settingsManager->setSetting('conversion', $conversionSettings);

            $this->notifySuccess(__('Currency settings updated successfully!'));
        } catch (\Exception $e) {
            $this->notifyError(__('Failed to update currency settings: ' . $e->getMessage()));
        }
    }

    protected function view($data = [])
    {
        return app('view')->file('/home/ziidfelj/home.ziidi.trade/storage/framework/views/livewire/views/55415330.blade.php', $data);
    }
}; 