<?php

use Livewire\Component;
use Livewire\Attributes\Validate;
use Livewire\Attributes\Computed;
use App\Traits\Vrm\Livewire\WithNotifications;
use Vormia\ATUMultiCurrency\Models\Currency;
use Vormia\ATUMultiCurrency\Support\CurrencySyncService;

return new class extends Component {
    use WithNotifications;

    // Currency ID
    public $currency_id;

    // Currency data
    public $currency;

    // Form fields
    #[Validate('nullable|string|min:3|max:4')]
    public $code = '';

    #[Validate('nullable|string|max:10')]
    public $symbol = '';

    #[Validate('nullable|string|max:255')]
    public $name = '';

    #[Validate('required|numeric|min:0.00000001')]
    public $rate = 1.0;

    #[Validate('required|boolean')]
    public $is_auto = false;

    #[Validate('nullable|numeric|min:0')]
    public $fee = null;

    #[Validate('nullable|integer|exists:vrm_taxonomies,id')]
    public $country_taxonomy_id = null;

    public function mount($id): void
    {
        $this->currency_id = $id;
        $this->currency = Currency::find($this->currency_id);

        if ($this->currency) {
            $this->code = $this->currency->code;
            $this->symbol = $this->currency->symbol;
            $this->name = $this->currency->name ?? '';
            $this->rate = $this->currency->rate;
            $this->is_auto = $this->currency->is_auto;
            $this->fee = $this->currency->fee;
            $this->country_taxonomy_id = $this->currency->country_taxonomy_id;
        }
    }

    #[Computed]
    public function country_list()
    {
        return \App\Models\Vrm\Taxonomy::where('group', 'country')->where('is_active', true)->get();
    }

    #[Computed]
    public function default_currency()
    {
        return Currency::where('is_default', true)->first();
    }

    // Update the Currency
    public function update(): void
    {
        // Apply automatic fallback logic: if code is empty, use symbol; if symbol is empty, use code
        if (empty(trim($this->code)) && !empty(trim($this->symbol))) {
            $this->code = $this->symbol;
        } elseif (empty(trim($this->symbol)) && !empty(trim($this->code))) {
            $this->symbol = $this->code;
        }

        // Validate that at least one of code or symbol is provided
        if (empty(trim($this->code)) && empty(trim($this->symbol))) {
            $this->notifyError(__('Either Currency Code or Currency Symbol must be provided.'));
            return;
        }

        // Validate code length after fallback
        if (!empty(trim($this->code))) {
            $codeLength = strlen(trim($this->code));
            if ($codeLength < 3 || $codeLength > 4) {
                $this->notifyError(__('Currency Code must be between 3 and 4 characters.'));
                return;
            }
        }

        // Custom validation for code uniqueness (excluding current currency)
        $this->validate([
            'code' => 'nullable|string|min:3|max:4',
            'symbol' => 'nullable|string|max:10',
            'name' => 'nullable|string|max:255',
            'rate' => 'required|numeric|min:0.00000001',
            'is_auto' => 'required|boolean',
            'fee' => 'nullable|numeric|min:0',
            'country_taxonomy_id' => 'nullable|integer|exists:vrm_taxonomies,id',
        ]);

        // Check if code already exists (excluding current currency)
        $codeExists = Currency::where('code', strtoupper($this->code))
            ->where('id', '!=', $this->currency_id)
            ->exists();

        if ($codeExists) {
            $this->notifyError(__('Currency code already exists.'));
            return;
        }

        // If default currency, rate must be 1.0
        if ($this->currency->is_default && $this->rate != 1.0) {
            $this->notifyError(__('Default currency rate must be 1.0.'));
            return;
        }

        try {
            $this->currency->update([
                'code' => strtoupper(trim($this->code)),
                'symbol' => trim($this->symbol),
                'name' => !empty(trim($this->name)) ? trim($this->name) : null,
                'rate' => $this->currency->is_default ? 1.0 : $this->rate,
                'is_auto' => $this->is_auto,
                'fee' => $this->fee,
                'country_taxonomy_id' => $this->country_taxonomy_id,
            ]);

            // If this is the default currency, sync with a2_ec_settings
            if ($this->currency->is_default) {
                $syncService = app(CurrencySyncService::class);
                $syncService->syncToA2Commerce();
            }

            // Refresh currency data
            $this->currency->refresh();
            $this->rate = $this->currency->rate;

            $this->notifySuccess(__('Currency updated successfully!'));
        } catch (\Exception $e) {
            $this->notifyError(__('Failed to update currency. Please try again: ' . $e->getMessage()));
        }
    }

    // Cancel
    public function cancel()
    {
        $this->notifyInfo(__('Currency update cancelled!'));
    }

    protected function view($data = [])
    {
        return app('view')->file('/home1/xxwewbmy/zidii.fxinvest.io/storage/framework/views/livewire/views/8bee0fea.blade.php', $data);
    }

    public function scriptModuleSrc()
    {
        return '/home1/xxwewbmy/zidii.fxinvest.io/storage/framework/views/livewire/scripts/8bee0fea.js';
    }
}; 