<?php

use App\Services\Trading\TradingSettingsService;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Component;

return new class extends Component {
    use WithNotifications;

    public bool $marketOpen = true;

    public string $mpesaWithdrawalMinAmount = '0';

    public string $mpesaWithdrawalMaxAmount = '0';

    public string $bankWithdrawalMinAmount = '0';

    public string $bankWithdrawalMaxAmount = '0';

    public string $sellMinSharesPerRequest = '1';

    public string $sellMaxSharesPerRequest = '0';

    public string $sellMaxSharesPerDay = '0';

    public string $depositMinAmount = '100';

    public string $depositMaxAmount = '0';

    public function mount(TradingSettingsService $settings): void
    {
        $this->marketOpen = $settings->isMarketOpen();
        $this->mpesaWithdrawalMinAmount = (string) $settings->withdrawalMinAmount('mpesa');
        $mpesaMax = $settings->withdrawalMaxAmount('mpesa');
        $this->mpesaWithdrawalMaxAmount = $mpesaMax >= PHP_FLOAT_MAX ? '0' : (string) $mpesaMax;

        $this->bankWithdrawalMinAmount = (string) $settings->withdrawalMinAmount('bank');
        $bankMax = $settings->withdrawalMaxAmount('bank');
        $this->bankWithdrawalMaxAmount = $bankMax >= PHP_FLOAT_MAX ? '0' : (string) $bankMax;

        $this->sellMinSharesPerRequest = (string) $settings->sellMinSharesPerRequest();
        $this->sellMaxSharesPerRequest = (string) $settings->sellMaxSharesPerRequest();
        $this->sellMaxSharesPerDay = (string) $settings->sellMaxSharesPerDay();

        $this->depositMinAmount = (string) $settings->depositMinAmount();
        $depositMax = $settings->depositMaxAmount();
        $this->depositMaxAmount = $depositMax >= PHP_FLOAT_MAX ? '0' : (string) $depositMax;
    }

    public function save(TradingSettingsService $settings): void
    {
        $validated = $this->validate([
            'marketOpen' => 'required|boolean',
            'mpesaWithdrawalMinAmount' => 'required|numeric|min:0',
            'mpesaWithdrawalMaxAmount' => 'required|numeric|min:0',
            'bankWithdrawalMinAmount' => 'required|numeric|min:0',
            'bankWithdrawalMaxAmount' => 'required|numeric|min:0',
            'sellMinSharesPerRequest' => 'required|integer|min:1',
            'sellMaxSharesPerRequest' => 'required|integer|min:0',
            'sellMaxSharesPerDay' => 'required|integer|min:0',
            'depositMinAmount' => 'required|numeric|min:0',
            'depositMaxAmount' => 'required|numeric|min:0',
        ]);

        if ((float) $validated['mpesaWithdrawalMaxAmount'] > 0 && (float) $validated['mpesaWithdrawalMaxAmount'] < (float) $validated['mpesaWithdrawalMinAmount']) {
            $this->addError('mpesaWithdrawalMaxAmount', __('M-Pesa withdrawal maximum amount must be greater than or equal to minimum amount.'));

            return;
        }

        if ((float) $validated['bankWithdrawalMaxAmount'] > 0 && (float) $validated['bankWithdrawalMaxAmount'] < (float) $validated['bankWithdrawalMinAmount']) {
            $this->addError('bankWithdrawalMaxAmount', __('Bank withdrawal maximum amount must be greater than or equal to minimum amount.'));

            return;
        }

        if ((int) $validated['sellMaxSharesPerRequest'] > 0 && (int) $validated['sellMaxSharesPerRequest'] < (int) $validated['sellMinSharesPerRequest']) {
            $this->addError('sellMaxSharesPerRequest', __('Sell maximum shares per request must be greater than or equal to minimum shares.'));

            return;
        }

        if ((int) $validated['sellMaxSharesPerDay'] > 0 && (int) $validated['sellMaxSharesPerDay'] < (int) $validated['sellMinSharesPerRequest']) {
            $this->addError('sellMaxSharesPerDay', __('Daily sell maximum must be greater than or equal to minimum shares per request.'));

            return;
        }

        if ((float) $validated['depositMaxAmount'] > 0 && (float) $validated['depositMaxAmount'] < (float) $validated['depositMinAmount']) {
            $this->addError('depositMaxAmount', __('Deposit maximum amount must be greater than or equal to minimum amount.'));

            return;
        }

        $settings->updateSettings(
            (bool) $validated['marketOpen'],
            $settings->investmentDurationDays(),
            $settings->investmentReturnPercent(),
        );
        $settings->updateWithdrawalLimits(
            (float) $validated['mpesaWithdrawalMinAmount'],
            (float) $validated['mpesaWithdrawalMaxAmount'],
            (float) $validated['bankWithdrawalMinAmount'],
            (float) $validated['bankWithdrawalMaxAmount'],
        );
        $settings->updateSellLimits(
            (int) $validated['sellMinSharesPerRequest'],
            (int) $validated['sellMaxSharesPerRequest'],
            (int) $validated['sellMaxSharesPerDay'],
        );
        $settings->updateDepositLimits(
            (float) $validated['depositMinAmount'],
            (float) $validated['depositMaxAmount'],
        );

        $this->notifySuccess(__('Settings updated successfully.'));
    }

    protected function view($data = [])
    {
        return app('view')->file('/home/ziidfelj/home.ziidi.trade/storage/framework/views/livewire/views/a6e46b17.blade.php', $data);
    }
}; 