<?php

use App\Concerns\PasswordValidationRules;
use App\Concerns\ProfileValidationRules;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Notifications\PasswordUpdatedNotification;
use App\Traits\Vrm\Livewire\WithNotifications;
use Illuminate\Support\Facades\Hash;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Component;

return new #[Layout('layouts.trading')] class extends Component {
    use WithNotifications;
    use PasswordValidationRules;
    use ProfileValidationRules;

    public string $depositAmount = '';
    public string $withdrawAmount = '';
    public string $name = '';
    public string $email = '';
    public string $current_password = '';
    public string $password = '';
    public string $password_confirmation = '';

    public function mount(): void
    {
        $this->name = auth()->user()->name;
        $this->email = auth()->user()->email;
    }

    #[Computed]
    public function wallet(): Wallet
    {
        return Wallet::query()->firstOrCreate(['user_id' => auth()->id()], ['currency_code' => 'USD']);
    }

    #[Computed]
    public function transactions()
    {
        return WalletTransaction::query()
            ->where('user_id', auth()->id())
            ->latest()
            ->limit(10)
            ->get();
    }

    public function deposit(): void
    {
        $amount = (float) $this->depositAmount;

        if ($amount <= 0) {
            $this->notifyError(__('Enter a valid deposit amount.'));

            return;
        }

        $wallet = $this->wallet;

        WalletTransaction::query()->create([
            'wallet_id' => $wallet->id,
            'user_id' => auth()->id(),
            'type' => 'deposit_request',
            'status' => 'pending',
            'amount' => $amount,
            'currency_code' => $wallet->currency_code,
            'description' => 'Wallet deposit request',
        ]);

        $this->depositAmount = '';
        $this->notifySuccess(__('Deposit request submitted. Awaiting admin approval.'));
    }

    public function withdraw(): void
    {
        $amount = (float) $this->withdrawAmount;

        if ($amount <= 0) {
            $this->notifyError(__('Enter a valid withdrawal amount.'));

            return;
        }

        $wallet = $this->wallet;

        if ($wallet->available_balance < $amount) {
            $this->notifyError(__('Insufficient available balance for this withdrawal.'));

            return;
        }

        $wallet->used_amount = (float) $wallet->used_amount + $amount;
        $wallet->save();

        WalletTransaction::query()->create([
            'wallet_id' => $wallet->id,
            'user_id' => auth()->id(),
            'type' => 'withdrawal_request',
            'status' => 'pending',
            'amount' => $amount,
            'currency_code' => $wallet->currency_code,
            'description' => 'Wallet withdrawal request',
        ]);

        $this->withdrawAmount = '';
        $this->notifySuccess(__('Withdrawal request submitted. Awaiting admin approval.'));
    }

    public function updateProfileInformation(): void
    {
        $user = auth()->user();
        $validated = $this->validate($this->profileRules($user->id));

        $user->fill($validated);

        if ($user->isDirty('email')) {
            $user->email_verified_at = null;
            $user->save();
            $user->sendEmailVerificationNotification();
            $this->notifySuccess(__('Profile updated. Please verify your new email address.'));

            return;
        }

        $user->save();
        $this->notifySuccess(__('Profile updated successfully.'));
    }

    public function updatePassword(): void
    {
        $validated = $this->validate([
            'current_password' => ['required', 'string'],
            'password' => $this->passwordRules(),
        ]);

        $user = auth()->user();

        if (! Hash::check($validated['current_password'], $user->password)) {
            $this->addError('current_password', __('The current password is incorrect.'));

            return;
        }

        $user->update([
            'password' => $validated['password'],
        ]);

        $user->notify(new PasswordUpdatedNotification());

        $this->reset('current_password', 'password', 'password_confirmation');
        $this->notifySuccess(__('Password updated successfully.'));
    }

    protected function view($data = [])
    {
        return app('view')->file('/Users/cybertruck/DevProjects/2026/FxZidii/storage/framework/views/livewire/views/aaa296ae.blade.php', $data);
    }
}; 