<?php

use App\Models\PortfolioHolding;
use App\Models\Security;
use App\Models\SecurityOrder;
use App\Models\SecurityWatchlist;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Services\Trading\TradingSettingsService;
use App\Traits\Vrm\Livewire\WithNotifications;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Component;

return new #[Layout('layouts.trading')] class extends Component {
    use WithNotifications;

    public string $search = '';
    public string $activeFilter = 'All';
    public bool $showWatchlist = false;
    public array $orderQuantities = [];
    public array $orderAmounts = [];
    public array $sellQuantities = [];
    public ?int $selectedSecurityId = null;

    public function mount(): void
    {
        $this->selectedSecurityId = Security::query()
            ->where('is_active', true)
            ->orderBy('trading_name')
            ->value('id');
    }

    #[Computed]
    public function wallet(): Wallet
    {
        return Wallet::query()->firstOrCreate(
            ['user_id' => auth()->id()],
            ['currency_code' => 'USD']
        );
    }

    #[Computed]
    public function isMarketOpen(): bool
    {
        return app(TradingSettingsService::class)->isMarketOpen();
    }

    #[Computed]
    public function securities()
    {
        return Security::query()
            ->where('is_active', true)
            ->orderBy('trading_name')
            ->get();
    }

    #[Computed]
    public function watchlistIds(): array
    {
        return auth()->user()->watchlistItems()->pluck('security_id')->all();
    }

    #[Computed]
    public function filteredSecurities()
    {
        $securities = $this->securities->filter(function (Security $security): bool {
            if ($this->search === '') {
                return true;
            }

            $query = strtolower($this->search);

            return str_contains(strtolower($security->trading_name), $query)
                || str_contains(strtolower($security->name), $query);
        });

        if ($this->activeFilter === 'Gainers') {
            $securities = $securities->filter(fn (Security $security): bool => $security->latestChangePercent() > 0);
        }

        if ($this->activeFilter === 'Losers') {
            $securities = $securities->filter(fn (Security $security): bool => $security->latestChangePercent() < 0);
        }

        return $securities->values();
    }

    #[Computed]
    public function watchlistSecurities()
    {
        return Security::query()
            ->whereIn('id', $this->watchlistIds)
            ->where('is_active', true)
            ->orderBy('trading_name')
            ->get();
    }

    #[Computed]
    public function selectedSecurity(): ?Security
    {
        if ($this->selectedSecurityId === null) {
            return null;
        }

        return Security::query()
            ->where('is_active', true)
            ->find($this->selectedSecurityId);
    }

    #[Computed]
    public function selectedHolding(): array
    {
        if ($this->selectedSecurityId === null) {
            return [
                'quantity' => 0,
                'reserved_quantity' => 0,
                'available_quantity' => 0,
                'average_buy_price' => 0,
                'total_invested' => 0,
                'current_value' => 0,
                'gain_loss_amount' => 0,
                'gain_loss_percent' => 0,
            ];
        }

        $holding = PortfolioHolding::query()
            ->where('user_id', auth()->id())
            ->where('security_id', $this->selectedSecurityId)
            ->first();

        if (! $holding || ! $this->selectedSecurity) {
            return [
                'quantity' => 0,
                'reserved_quantity' => 0,
                'available_quantity' => 0,
                'average_buy_price' => 0,
                'total_invested' => 0,
                'current_value' => 0,
                'gain_loss_amount' => 0,
                'gain_loss_percent' => 0,
            ];
        }

        $quantity = (int) $holding->quantity;
        $reservedQuantity = (int) ($holding->reserved_quantity ?? 0);
        $availableQuantity = max(0, $quantity - $reservedQuantity);
        $averageBuyPrice = (float) $holding->average_buy_price;
        $totalInvested = (float) $holding->total_invested;
        $currentValue = (float) $this->selectedSecurity->current_amount * $quantity;
        $gainLossAmount = $currentValue - $totalInvested;
        $gainLossPercent = $totalInvested > 0 ? ($gainLossAmount / $totalInvested) * 100 : 0;

        return [
            'quantity' => $quantity,
            'reserved_quantity' => $reservedQuantity,
            'available_quantity' => $availableQuantity,
            'average_buy_price' => $averageBuyPrice,
            'total_invested' => $totalInvested,
            'current_value' => $currentValue,
            'gain_loss_amount' => $gainLossAmount,
            'gain_loss_percent' => $gainLossPercent,
        ];
    }

    public function setFilter(string $filter): void
    {
        $this->activeFilter = $filter;
    }

    public function toggleWatchlist(): void
    {
        $this->showWatchlist = ! $this->showWatchlist;
    }

    public function selectSecurity(int $securityId): void
    {
        $security = Security::query()
            ->where('is_active', true)
            ->find($securityId);

        if (! $security) {
            $this->notifyError(__('Security not found.'));

            return;
        }

        $this->selectedSecurityId = $securityId;
    }

    public function toggleSecurityWatchlist(int $securityId): void
    {
        $existing = SecurityWatchlist::query()
            ->where('user_id', auth()->id())
            ->where('security_id', $securityId)
            ->first();

        if ($existing) {
            $existing->delete();
            $this->notifyInfo(__('Removed from watchlist.'));

            return;
        }

        SecurityWatchlist::query()->create([
            'user_id' => auth()->id(),
            'security_id' => $securityId,
        ]);

        $this->notifySuccess(__('Added to watchlist.'));
    }

    public function placeOrder(int $securityId): void
    {
        $security = Security::query()->find($securityId);

        if (! $security) {
            $this->notifyError(__('Security not found.'));

            return;
        }

        $quantity = (int) ($this->orderQuantities[$securityId] ?? 0);

        $this->submitOrder($security, $quantity, 'buy');
    }

    public function placeBuyOrder(int $securityId): void
    {
        $security = Security::query()->find($securityId);

        if (! $security) {
            $this->notifyError(__('Security not found.'));

            return;
        }

        $amount = (float) ($this->orderAmounts[$securityId] ?? 0);

        if ($amount <= 0) {
            $this->notifyError(__('Enter a valid amount to invest.'));

            return;
        }

        $quantity = (int) floor($amount / (float) $security->current_amount);

        if ($quantity <= 0) {
            $this->notifyError(__('Amount is too low for this security price.'));

            return;
        }

        $this->submitOrder($security, $quantity, 'buy');
    }

    public function placeSellOrder(int $securityId): void
    {
        $security = Security::query()->find($securityId);

        if (! $security) {
            $this->notifyError(__('Security not found.'));

            return;
        }

        $quantity = (int) ($this->sellQuantities[$securityId] ?? 0);

        $this->submitOrder($security, $quantity, 'sell');
    }

    protected function submitOrder(Security $security, int $quantity, string $orderType): void
    {
        if (! $this->isMarketOpen) {
            $this->notifyError(__('Market is currently closed. Trading is unavailable.'));

            return;
        }

        if ($quantity < $security->min_shares) {
            $this->notifyError(__('Quantity must be at least :min shares.', ['min' => $security->min_shares]));

            return;
        }

        if ($security->max_shares_purchase > 0 && $quantity > $security->max_shares_purchase) {
            $this->notifyError(__('Quantity exceeds max shares per purchase.'));

            return;
        }

        $wallet = $this->wallet;
        $totalAmount = (float) $security->current_amount * $quantity;

        if ($orderType === 'buy') {
            $existingHoldingQuantity = (int) auth()->user()
                ->portfolioHoldings()
                ->where('security_id', $security->id)
                ->value('quantity');

            if ($security->max_shares_holding > 0 && ($existingHoldingQuantity + $quantity) > $security->max_shares_holding) {
                $this->notifyError(__('Order exceeds your maximum allowed holding for this security.'));

                return;
            }

            if ($wallet->available_balance < $totalAmount) {
                $this->notifyError(__('Insufficient available balance.'));

                return;
            }

            DB::transaction(function () use ($security, $quantity, $wallet, $totalAmount): void {
                SecurityOrder::query()->create([
                    'user_id' => auth()->id(),
                    'security_id' => $security->id,
                    'wallet_id' => $wallet->id,
                    'order_type' => 'buy',
                    'status' => 'pending',
                    'price_per_share' => $security->current_amount,
                    'quantity' => $quantity,
                    'volume' => $quantity,
                    'total_amount' => $totalAmount,
                    'currency_code' => $wallet->currency_code,
                    'traded_at' => now(),
                    'profit_loss_amount' => 0,
                    'profit_loss_percent' => 0,
                    'is_active' => true,
                ]);

                $wallet->used_amount = (float) $wallet->used_amount + $totalAmount;
                $wallet->save();

                WalletTransaction::query()->create([
                    'wallet_id' => $wallet->id,
                    'user_id' => auth()->id(),
                    'type' => 'order_debit',
                    'status' => 'pending',
                    'amount' => $totalAmount,
                    'currency_code' => $wallet->currency_code,
                    'description' => "Pending buy order for {$security->trading_name}",
                    'meta' => ['security_id' => $security->id, 'quantity' => $quantity, 'order_type' => 'buy'],
                ]);
            });

            $this->orderQuantities[$security->id] = 0;
            $this->orderAmounts[$security->id] = '';
            $this->notifySuccess(__('Buy order submitted and pending admin approval.'));

            return;
        }

        try {
            DB::transaction(function () use ($security, $quantity, $wallet, $totalAmount): void {
                $holding = PortfolioHolding::query()
                    ->where('user_id', auth()->id())
                    ->where('security_id', $security->id)
                    ->lockForUpdate()
                    ->first();

                if (! $holding || (int) $holding->quantity <= 0) {
                    throw new RuntimeException(__('You do not have this security in your portfolio.'));
                }

                $availableQuantity = max(0, (int) $holding->quantity - (int) ($holding->reserved_quantity ?? 0));

                if ($availableQuantity < $quantity) {
                    throw new RuntimeException(__('Sell quantity exceeds your available shares.'));
                }

                $holding->reserved_quantity = (int) ($holding->reserved_quantity ?? 0) + $quantity;
                $holding->save();

                SecurityOrder::query()->create([
                    'user_id' => auth()->id(),
                    'security_id' => $security->id,
                    'wallet_id' => $wallet->id,
                    'order_type' => 'sell',
                    'status' => 'pending',
                    'price_per_share' => $security->current_amount,
                    'quantity' => $quantity,
                    'volume' => $quantity,
                    'total_amount' => $totalAmount,
                    'currency_code' => $wallet->currency_code,
                    'traded_at' => now(),
                    'profit_loss_amount' => 0,
                    'profit_loss_percent' => 0,
                    'is_active' => false,
                ]);

                WalletTransaction::query()->create([
                    'wallet_id' => $wallet->id,
                    'user_id' => auth()->id(),
                    'type' => 'order_credit',
                    'status' => 'pending',
                    'amount' => $totalAmount,
                    'currency_code' => $wallet->currency_code,
                    'description' => "Pending sell order for {$security->trading_name}",
                    'meta' => ['security_id' => $security->id, 'quantity' => $quantity, 'order_type' => 'sell'],
                ]);
            });
        } catch (RuntimeException $exception) {
            $this->notifyError($exception->getMessage());

            return;
        }

        $this->sellQuantities[$security->id] = 0;
        $this->notifySuccess(__('Sell order submitted and pending admin approval.'));
    }

    protected function view($data = [])
    {
        return app('view')->file('/Users/cybertruck/DevProjects/2026/FxZidii/storage/framework/views/livewire/views/c1b724d7.blade.php', $data);
    }
}; 