<?php

use App\Facades\Vrm\MediaForge;
use App\Models\InvestmentPackage;
use App\Traits\Vrm\Livewire\WithNotifications;
use Livewire\Component;
use Livewire\WithFileUploads;

return new class extends Component {
    use WithFileUploads;
    use WithNotifications;

    public int $packageId;

    public string $name = '';

    public string $min_amount = '';

    public string $max_amount = '';

    public string $interest_percent = '';

    public string $duration_hours = '';

    public string $description = '';

    public $photo = null;

    public string $currency_code = '';

    public bool $is_active = true;

    public function mount(int $id): void
    {
        $this->packageId = $id;
        $package = InvestmentPackage::query()->find($id);
        if (! $package) {
            $this->notifyError(__('Package not found.'));
            $this->redirect(route('admin.investment-packages.index'), navigate: true);

            return;
        }
        $this->name = $package->name;
        $this->min_amount = (string) $package->min_amount;
        $this->max_amount = (string) $package->max_amount;
        $this->interest_percent = (string) $package->interest_percent;
        $this->duration_hours = (string) $package->duration_hours;
        $this->description = (string) $package->description;
        $this->currency_code = (string) ($package->currency_code ?? '');
        $this->is_active = $package->is_active;
    }

    public function save(): void
    {
        $this->validate([
            'name' => 'required|string|max:255',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'required|numeric|min:0|gte:min_amount',
            'interest_percent' => 'required|numeric|min:0|max:10000',
            'duration_hours' => 'required|integer|min:1',
            'description' => 'nullable|string|max:5000',
            'photo' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'currency_code' => 'nullable|string|size:3',
        ], [
            'max_amount.gte' => __('Max amount must be greater than or equal to min amount.'),
        ]);

        $package = InvestmentPackage::query()->find($this->packageId);
        if (! $package) {
            $this->notifyError(__('Package not found.'));

            return;
        }

        $photoPath = $package->photo;
        if ($this->photo) {
            $photoPath = MediaForge::upload($this->photo)
                ->useYearFolder(true)
                ->randomizeFileName(true)
                ->to('investment-packages')
                ->resize(300, 300)
                ->run();
        }

        $package->update([
            'name' => $this->name,
            'min_amount' => (float) $this->min_amount,
            'max_amount' => (float) $this->max_amount,
            'interest_percent' => (float) $this->interest_percent,
            'duration_hours' => (int) $this->duration_hours,
            'description' => $this->description ?: null,
            'photo' => $photoPath,
            'is_active' => $this->is_active,
            'currency_code' => $this->currency_code ?: null,
        ]);

        $this->notifySuccess(__('Investment package updated successfully.'));
        $this->redirect(route('admin.investment-packages.index'), navigate: true);
    }

    protected function view($data = [])
    {
        return app('view')->file('/home1/xxwewbmy/zidii.fxinvest.io/storage/framework/views/livewire/views/c559c2e2.blade.php', $data);
    }
}; 