<?php

use App\Models\User;
use App\Models\Vrm\Role;
use Laravel\Fortify\Features;

test('login screen can be rendered', function () {
    $response = $this->get(route('login'));

    $response->assertOk();
});

test('users can authenticate using the login screen', function () {
    $memberRole = Role::query()->firstOrCreate(
        ['id' => 3],
        [
            'name' => 'Member',
            'slug' => 'member',
            'module' => 'contribution',
            'authority' => 'member',
            'is_active' => true,
            'description' => 'Member',
        ]
    );

    $user = User::factory()->create();
    $user->roles()->attach($memberRole->id);

    $response = $this->post(route('login.store'), [
        'email' => $user->email,
        'password' => 'password',
    ]);

    $response
        ->assertSessionHasNoErrors()
        ->assertRedirect(route('member.dashboard', absolute: false));

    $this->assertAuthenticated();
});

test('unverified members are redirected to email verification notice after login', function () {
    $memberRole = Role::query()->firstOrCreate(
        ['id' => 3],
        [
            'name' => 'Member',
            'slug' => 'member',
            'module' => 'contribution',
            'authority' => 'member',
            'is_active' => true,
            'description' => 'Member',
        ]
    );

    $user = User::factory()->unverified()->create();
    $user->roles()->attach($memberRole->id);

    $response = $this->post(route('login.store'), [
        'email' => $user->email,
        'password' => 'password',
    ]);

    $response
        ->assertSessionHasNoErrors()
        ->assertRedirect(route('verification.notice', absolute: false));

    $this->assertAuthenticated();
});

test('users can not authenticate with invalid password', function () {
    $user = User::factory()->create();

    $response = $this->post(route('login.store'), [
        'email' => $user->email,
        'password' => 'wrong-password',
    ]);

    $response->assertSessionHasErrorsIn('email');

    $this->assertGuest();
});

test('inactive users cannot authenticate', function () {
    $user = User::factory()->inactive()->create();

    $response = $this->post(route('login.store'), [
        'email' => $user->email,
        'password' => 'password',
    ]);

    $response->assertSessionHasErrorsIn('email');
    $this->assertGuest();
});

test('users with two factor enabled are redirected to two factor challenge', function () {
    if (! Features::canManageTwoFactorAuthentication()) {
        $this->markTestSkipped('Two-factor authentication is not enabled.');
    }

    Features::twoFactorAuthentication([
        'confirm' => true,
        'confirmPassword' => true,
    ]);

    $user = User::factory()->withTwoFactor()->create();

    $response = $this->post(route('login.store'), [
        'email' => $user->email,
        'password' => 'password',
    ]);

    $response->assertRedirect(route('two-factor.login'));
    $this->assertGuest();
});

test('users can logout', function () {
    $user = User::factory()->create();

    $response = $this->actingAs($user)->post(route('logout'));

    $response->assertRedirect(route('home'));

    $this->assertGuest();
});
