<?php

use App\Models\Investment;
use App\Models\User;
use App\Models\Vrm\Role;
use App\Models\Vrm\Utility;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use Livewire\Livewire;

function ensureRoleForInvestment(int $id, string $name, string $slug): Role
{
    return Role::query()->firstOrCreate(
        ['id' => $id],
        [
            'name' => $name,
            'slug' => $slug,
            'module' => 'Dashboard',
            'authority' => $slug,
            'is_active' => true,
            'description' => $name,
        ]
    );
}

test('member can create investment and claim double return after maturity', function () {
    $memberRole = ensureRoleForInvestment(3, 'Member', 'member');

    $member = User::factory()->create();
    $member->roles()->attach($memberRole->id);

    $wallet = Wallet::query()->create([
        'user_id' => $member->id,
        'currency_code' => 'USD',
        'balance' => 1000,
        'used_amount' => 0,
        'earned_amount' => 0,
    ]);

    $this->actingAs($member);

    Utility::query()->updateOrCreate(
        ['type' => 'general', 'key' => 'investment_duration_days'],
        ['value' => '2']
    );
    Utility::query()->updateOrCreate(
        ['type' => 'general', 'key' => 'investment_return_percent'],
        ['value' => '100']
    );

    Livewire::test('front.investments')
        ->set('amount', '100')
        ->call('invest');

    $investment = Investment::query()->where('user_id', $member->id)->latest()->first();
    expect($investment)->not()->toBeNull();
    expect($investment->status)->toBe('active');
    expect((float) $investment->principal_amount)->toBe(100.0);
    expect((float) $investment->expected_return_amount)->toBe(200.0);
    expect($investment->duration_days)->toBe(2);

    $wallet->refresh();
    expect((float) $wallet->balance)->toBe(900.0);

    $debit = WalletTransaction::query()->where('type', 'investment_debit')->first();
    expect($debit)->not()->toBeNull();
    expect((float) $debit->amount)->toBe(100.0);

    $this->travel(3)->days();

    Livewire::test('front.investments')
        ->call('claim', $investment->id);

    $investment->refresh();
    expect($investment->status)->toBe('claimed');
    expect($investment->is_active)->toBeFalse();
    expect($investment->claimed_at)->not()->toBeNull();

    $wallet->refresh();
    expect((float) $wallet->balance)->toBe(1100.0);
    expect((float) $wallet->earned_amount)->toBe(100.0);

    $credit = WalletTransaction::query()->where('type', 'investment_credit')->first();
    expect($credit)->not()->toBeNull();
    expect((float) $credit->amount)->toBe(200.0);
});
