<?php

use App\Models\User;
use App\Models\Vrm\Role;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use Livewire\Livewire;

function ensureRoleForMpesa(int $id, string $name, string $slug): Role
{
    return Role::query()->firstOrCreate(
        ['id' => $id],
        [
            'name' => $name,
            'slug' => $slug,
            'module' => 'Dashboard',
            'authority' => $slug,
            'is_active' => true,
            'description' => $name,
        ]
    );
}

test('member can submit mpesa deposit request and admin can approve it', function () {
    $adminRole = ensureRoleForMpesa(2, 'Admin', 'admin');
    $memberRole = ensureRoleForMpesa(3, 'Member', 'member');

    $admin = User::factory()->create();
    $admin->roles()->attach($adminRole->id);

    $member = User::factory()->create();
    $member->roles()->attach($memberRole->id);

    $wallet = Wallet::query()->create([
        'user_id' => $member->id,
        'currency_code' => 'USD',
        'balance' => 100,
        'used_amount' => 0,
        'earned_amount' => 0,
    ]);

    $this->actingAs($member);

    Livewire::test('front.mpesa-deposit')
        ->set('amount', '250')
        ->set('phoneNumber', '254712345678')
        ->set('mpesaReference', 'QWE123ABC')
        ->call('submitDepositRequest');

    $request = WalletTransaction::query()->where('user_id', $member->id)->latest()->first();
    expect($request)->not()->toBeNull();
    expect($request->type)->toBe('mpesa_deposit_request');
    expect($request->status)->toBe('pending');
    expect((float) $request->amount)->toBe(250.0);
    expect(data_get($request->meta, 'phone_number'))->toBe('254712345678');

    $this->actingAs($admin);

    Livewire::test('admin.control.wallet-requests.index')
        ->call('approve', $request->id);

    $request->refresh();
    expect($request->status)->toBe('approved');

    $wallet->refresh();
    expect((float) $wallet->balance)->toBe(350.0);
});
