<?php

use App\Models\Security;
use App\Services\LiveStockService;
use Illuminate\Support\Facades\Http;

beforeEach(function () {
    config([
        'services.rapidapi_nse.host' => 'nairobi-stock-exchange-nse.p.rapidapi.com',
        'services.rapidapi_nse.key' => 'test-key',
    ]);
});

test('getAllStocks returns decoded JSON from RapidNSE', function () {
    $payload = [
        'success' => true,
        'data' => [
            'data' => [
                ['ticker' => 'SCOM'],
                ['ticker' => 'KCB'],
            ],
        ],
    ];

    Http::fake([
        'https://nairobi-stock-exchange-nse.p.rapidapi.com/stocks' => Http::response($payload, 200),
    ]);

    $service = app(LiveStockService::class);
    $result = $service->getAllStocks();

    expect($result)->toBe($payload);
});

test('getSecuritiesForDisplay merges API data with Security id and logo', function () {
    $apiPayload = [
        'success' => true,
        'data' => [
            'data' => [
                ['ticker' => 'SCOM', 'name' => 'Safaricom PLC', 'price' => '15.50', 'change' => '+1.20'],
                ['ticker' => 'KCB', 'name' => 'KCB Group', 'price' => '42.00', 'change' => '-0.50'],
            ],
        ],
    ];

    Http::fake([
        'https://nairobi-stock-exchange-nse.p.rapidapi.com/stocks' => Http::response($apiPayload, 200),
    ]);

    $security = Security::query()->create([
        'name' => 'Safaricom PLC',
        'trading_name' => 'SCOM',
        'logo' => 'https://example.com/safaricom.png',
        'currency_code' => 'USD',
        'initial_listing_amount' => 10,
        'current_amount' => 15.50,
        'is_active' => true,
    ]);

    $service = app(LiveStockService::class);
    $result = $service->getSecuritiesForDisplay(10);

    expect($result)->toHaveCount(2);

    $safaricom = collect($result)->firstWhere('symbol', 'SCOM');
    expect($safaricom)->not->toBeNull()
        ->and($safaricom['id'])->toBe($security->id)
        ->and($safaricom['symbol'])->toBe('SCOM')
        ->and($safaricom['name'])->toBe('Safaricom PLC')
        ->and($safaricom['price'])->toBe(15.5)
        ->and($safaricom['change'])->toBe(1.2)
        ->and($safaricom['logo'])->toBe('https://example.com/safaricom.png')
        ->and($safaricom['initials'])->toBe('SC');

    $kcb = collect($result)->firstWhere('symbol', 'KCB');
    expect($kcb)->not->toBeNull()
        ->and($kcb['id'])->toBeNull()
        ->and($kcb['logo'])->toBeNull()
        ->and($kcb['price'])->toBe(42.0)
        ->and($kcb['change'])->toBe(-0.5)
        ->and($kcb['initials'])->toBe('KC');
});

test('getSecuritiesForDisplay respects limit', function () {
    $apiPayload = [
        'success' => true,
        'data' => [
            'data' => array_fill(0, 50, ['ticker' => 'STK', 'price' => '1', 'change' => '+0.00']),
        ],
    ];

    Http::fake([
        'https://nairobi-stock-exchange-nse.p.rapidapi.com/stocks' => Http::response($apiPayload, 200),
    ]);

    $service = app(LiveStockService::class);
    $result = $service->getSecuritiesForDisplay(5);

    expect($result)->toHaveCount(5);
});
