<?php

use App\Models\PortfolioHolding;
use App\Models\Security;
use App\Models\SecurityOrder;
use App\Models\User;
use App\Models\Vrm\Role;
use App\Models\Wallet;
use App\Models\WalletTransaction;
use App\Notifications\SecurityOrderStatusNotification;
use App\Notifications\WalletTransactionNotification;
use Illuminate\Support\Facades\Notification;
use Livewire\Livewire;

function ensureRole(int $id, string $name, string $slug): Role
{
    return Role::query()->firstOrCreate(
        ['id' => $id],
        [
            'name' => $name,
            'slug' => $slug,
            'module' => 'Dashboard',
            'authority' => $slug,
            'is_active' => true,
            'description' => $name,
        ]
    );
}

test('member can place pending buy order from trade screen', function () {
    $memberRole = ensureRole(3, 'Member', 'member');

    $member = User::factory()->create();
    $member->roles()->attach($memberRole->id);

    $security = Security::query()->forceCreate([
        'name' => 'Apple Inc',
        'trading_name' => 'AAPL',
        'currency_code' => 'USD',
        'initial_listing_amount' => 100,
        'current_amount' => 100,
        'min_shares' => 1,
        'max_shares_purchase' => 0,
        'max_shares_holding' => 0,
        'is_active' => true,
    ]);

    $wallet = Wallet::query()->create([
        'user_id' => $member->id,
        'currency_code' => 'USD',
        'balance' => 5000,
        'used_amount' => 0,
        'earned_amount' => 0,
    ]);

    $this->actingAs($member);

    Livewire::test('front.trade')
        ->set("orderQuantities.{$security->id}", 10)
        ->call('placeOrder', $security->id);

    expect(SecurityOrder::query()->where('user_id', $member->id)->count())->toBe(1);

    $order = SecurityOrder::query()->first();
    expect($order->status)->toBe('pending');
    expect((float) $order->total_amount)->toBe(1000.0);
    expect($order->volume)->toBe(10);
    expect($order->is_active)->toBeTrue();
    expect($order->traded_at)->not->toBeNull();
    expect((float) $order->profit_loss_amount)->toBe(0.0);
    expect((float) $order->profit_loss_percent)->toBe(0.0);

    $wallet->refresh();
    expect((float) $wallet->used_amount)->toBe(1000.0);

    $transaction = WalletTransaction::query()->first();
    expect($transaction->status)->toBe('pending');
    expect($transaction->type)->toBe('order_debit');
});

test('member can place pending sell order and reserved shares are tracked', function () {
    $memberRole = ensureRole(3, 'Member', 'member');

    $member = User::factory()->create();
    $member->roles()->attach($memberRole->id);

    $security = Security::query()->forceCreate([
        'name' => 'Safaricom PLC',
        'trading_name' => 'SCOM',
        'currency_code' => 'USD',
        'initial_listing_amount' => 120,
        'current_amount' => 120,
        'min_shares' => 1,
        'max_shares_purchase' => 0,
        'max_shares_holding' => 0,
        'is_active' => true,
    ]);

    Wallet::query()->create([
        'user_id' => $member->id,
        'currency_code' => 'USD',
        'balance' => 500,
        'used_amount' => 0,
        'earned_amount' => 0,
    ]);

    $holding = PortfolioHolding::query()->create([
        'user_id' => $member->id,
        'security_id' => $security->id,
        'quantity' => 10,
        'reserved_quantity' => 0,
        'average_buy_price' => 100,
        'total_invested' => 1000,
    ]);

    $this->actingAs($member);

    Livewire::test('front.trade')
        ->set("sellQuantities.{$security->id}", 4)
        ->call('placeSellOrder', $security->id);

    expect(SecurityOrder::query()->where('user_id', $member->id)->count())->toBe(1);

    $order = SecurityOrder::query()->first();
    expect($order->order_type)->toBe('sell');
    expect($order->status)->toBe('pending');
    expect((float) $order->total_amount)->toBe(480.0);

    $holding->refresh();
    expect($holding->quantity)->toBe(10);
    expect((int) $holding->reserved_quantity)->toBe(4);

    Livewire::test('front.trade')
        ->set("sellQuantities.{$security->id}", 7)
        ->call('placeSellOrder', $security->id);

    $holding->refresh();
    expect((int) $holding->reserved_quantity)->toBe(4);
    expect(SecurityOrder::query()->where('user_id', $member->id)->count())->toBe(1);

    $transaction = WalletTransaction::query()->where('user_id', $member->id)->first();
    expect($transaction)->not()->toBeNull();
    expect($transaction->status)->toBe('pending');
    expect($transaction->type)->toBe('order_credit');
});

test('admin can approve pending order and member receives email notification', function () {
    Notification::fake();

    $adminRole = ensureRole(2, 'Admin', 'admin');
    $memberRole = ensureRole(3, 'Member', 'member');

    $admin = User::factory()->create();
    $admin->roles()->attach($adminRole->id);

    $member = User::factory()->create();
    $member->roles()->attach($memberRole->id);

    $security = Security::query()->forceCreate([
        'name' => 'KCB Group PLC',
        'trading_name' => 'KCB',
        'currency_code' => 'USD',
        'initial_listing_amount' => 42.5,
        'current_amount' => 42.5,
        'min_shares' => 1,
        'max_shares_purchase' => 0,
        'max_shares_holding' => 0,
        'is_active' => true,
    ]);

    $wallet = Wallet::query()->create([
        'user_id' => $member->id,
        'currency_code' => 'USD',
        'balance' => 3000,
        'used_amount' => 425,
        'earned_amount' => 0,
    ]);

    $order = SecurityOrder::query()->create([
        'user_id' => $member->id,
        'security_id' => $security->id,
        'wallet_id' => $wallet->id,
        'order_type' => 'buy',
        'status' => 'pending',
        'price_per_share' => 42.5,
        'quantity' => 10,
        'volume' => 10,
        'total_amount' => 425,
        'currency_code' => 'USD',
        'traded_at' => now(),
        'profit_loss_amount' => 0,
        'profit_loss_percent' => 0,
        'is_active' => true,
    ]);

    $this->actingAs($admin);

    Livewire::test('admin.control.orders.index')
        ->call('approve', $order->id);

    $order->refresh();
    expect($order->status)->toBe('approved');
    expect($order->approved_by)->toBe($admin->id);
    expect($order->is_active)->toBeTrue();
    expect((float) $order->profit_loss_amount)->toBe(0.0);
    expect((float) $order->profit_loss_percent)->toBe(0.0);

    $wallet->refresh();
    expect((float) $wallet->used_amount)->toBe(0.0);
    expect((float) $wallet->balance)->toBe(2575.0);

    $holding = PortfolioHolding::query()->where('user_id', $member->id)->where('security_id', $security->id)->first();
    expect($holding)->not()->toBeNull();
    expect($holding->quantity)->toBe(10);

    Notification::assertSentTo($member, SecurityOrderStatusNotification::class);
});

test('admin can approve pending sell order and credits wallet', function () {
    Notification::fake();

    $adminRole = ensureRole(2, 'Admin', 'admin');
    $memberRole = ensureRole(3, 'Member', 'member');

    $admin = User::factory()->create();
    $admin->roles()->attach($adminRole->id);

    $member = User::factory()->create();
    $member->roles()->attach($memberRole->id);

    $security = Security::query()->forceCreate([
        'name' => 'Safaricom PLC',
        'trading_name' => 'SCOM',
        'currency_code' => 'USD',
        'initial_listing_amount' => 120,
        'current_amount' => 120,
        'min_shares' => 1,
        'max_shares_purchase' => 0,
        'max_shares_holding' => 0,
        'is_active' => true,
    ]);

    $wallet = Wallet::query()->create([
        'user_id' => $member->id,
        'currency_code' => 'USD',
        'balance' => 500,
        'used_amount' => 0,
        'earned_amount' => 0,
    ]);

    $holding = PortfolioHolding::query()->create([
        'user_id' => $member->id,
        'security_id' => $security->id,
        'quantity' => 10,
        'reserved_quantity' => 4,
        'average_buy_price' => 100,
        'total_invested' => 1000,
    ]);

    $order = SecurityOrder::query()->create([
        'user_id' => $member->id,
        'security_id' => $security->id,
        'wallet_id' => $wallet->id,
        'order_type' => 'sell',
        'status' => 'pending',
        'price_per_share' => 120,
        'quantity' => 4,
        'volume' => 4,
        'total_amount' => 480,
        'currency_code' => 'USD',
        'traded_at' => now(),
        'profit_loss_amount' => 0,
        'profit_loss_percent' => 0,
        'is_active' => false,
    ]);

    $this->actingAs($admin);

    Livewire::test('admin.control.orders.index')
        ->call('approve', $order->id);

    $order->refresh();
    expect($order->status)->toBe('approved');
    expect($order->approved_by)->toBe($admin->id);
    expect($order->is_active)->toBeFalse();
    expect((float) $order->profit_loss_amount)->toBe(80.0);
    expect((float) $order->profit_loss_percent)->toBe(20.0);

    $wallet->refresh();
    expect((float) $wallet->balance)->toBe(980.0);
    expect((float) $wallet->used_amount)->toBe(0.0);

    $holding->refresh();
    expect($holding->quantity)->toBe(6);
    expect((int) $holding->reserved_quantity)->toBe(0);
    expect((float) $holding->total_invested)->toBe(600.0);
    expect((float) $holding->average_buy_price)->toBe(100.0);

    Notification::assertSentTo($member, SecurityOrderStatusNotification::class);
});

test('member wallet deposit request can be approved by admin', function () {
    Notification::fake();

    $adminRole = ensureRole(2, 'Admin', 'admin');
    $memberRole = ensureRole(3, 'Member', 'member');

    $admin = User::factory()->create();
    $admin->roles()->attach($adminRole->id);

    $member = User::factory()->create();
    $member->roles()->attach($memberRole->id);

    Wallet::query()->create([
        'user_id' => $member->id,
        'currency_code' => 'USD',
        'balance' => 100,
        'used_amount' => 0,
        'earned_amount' => 0,
    ]);

    $this->actingAs($member);

    Livewire::test('front.account')
        ->set('depositAmount', '250')
        ->call('deposit');

    $request = WalletTransaction::query()->where('user_id', $member->id)->latest()->first();

    expect($request)->not()->toBeNull();
    expect($request->type)->toBe('deposit_request');
    expect($request->status)->toBe('pending');

    $this->actingAs($admin);

    Livewire::test('admin.control.wallet-requests.index')
        ->call('approve', $request->id);

    $request->refresh();
    expect($request->status)->toBe('approved');

    $wallet = Wallet::query()->where('user_id', $member->id)->first();
    expect((float) $wallet->balance)->toBe(350.0);

    Notification::assertSentTo($member, WalletTransactionNotification::class);
});
